/* IMSpector - Instant Messenger Transparent Proxy Service
 * http://www.imspector.org/
 * (c) Lawrence Manning <lawrence@aslak.net>, 2006
 * 
 * Contributions from:
 *     Ryan Wagoner <ryan@wgnrs.dynu.com>, 2006
 * 
 * Released under the GPL v2. */

#include "imspector.h"

#define DEFAULT_CONFIG "/usr/etc/imspector/imspector.conf"
#define LOGGING_SOCKET "/tmp/.imspectorlog"
#define DEFAULT_PLUGIN_DIR "/usr/lib/imspector"

#define DEFAULT_RESPONSE_PREFIX "Message from IMSpector: -="
#define DEFAULT_RESPONSE_POSTFIX "=-"

std::vector<class ProtocolPlugin> protocolplugins;
std::vector<class FilterPlugin> filterplugins;
std::vector<class ResponderPlugin> responderplugins;

volatile bool quitting = false;

bool handleserversock(class Options &options, bool debugmode,
	class SSLState &sslstate, class Socket &serversock, bool http);
bool doproxy(class Options &options, bool debugmode, class Socket &clientsock, std::string &clientaddress,
	class SSLState &sslstate, bool http);
bool connectsockethttperror(class Socket &imserversock, std::string redirectaddress,
	std::string interface, class Socket &clientsock, bool http);
bool sendhttpresponse(bool http, class Socket &sock, std::string response, std::string usermessage);
bool loggingprocess(class Options &options, bool debugmode, class Socket &logsock);
bool logimevents(std::vector<struct imevent> &imevents);
bool runfilterplugins(char *buffer, std::vector<struct imevent> &imevents);
void massageimevents(class Options &options, std::vector<struct imevent> &imevents);

int getgidfromname(std::string name);
int getuidfromname(std::string name);
bool checkforrunning(std::string pidfilename);
bool writepidfile(std::string pidfilename);
bool removepidfile(std::string pidfilename);

bool clearsighandlers(void);
void fireman(int signal);
void sigterm(int signal);

int main(int argc, char *argv[])
{
	std::string configfilename = DEFAULT_CONFIG;
	class Options options;
	bool debugmode = false;
	
	/* Parse arguments. */
	for (int c = 1; c < argc; c++)
	{
		/* Override the default config file. */
		if ((c + 1) < argc)
		{
			if (strcmp("-c", argv[c]) == 0)
			{
				configfilename = argv[++c];
				continue;
			}
		}
		/* Turn on debug mode. */
		if (strcmp("-d", argv[c]) == 0)
		{
			debugmode = true;
			continue;
		}
		/* Display usage. */
		if (strcmp("-h", argv[c]) == 0 || strcmp("--help", argv[c]) == 0)
		{
			fprintf(stderr, "Usage:\n" \
				"\t%s [-c configfile] [-d]\n" \
				"Notes:\n" \
				"\tWith -d IMSpector will run in debug mode\n",
				argv[0]);
			return 1;
		}		
		
		fprintf(stderr, "Warning: Unrecognised argument: %s\n", argv[c]);
	}

	if (!options.readoptionsfile(configfilename))
	{
		fprintf(stderr, "Coudln't read options file %s\n", configfilename.c_str());
		return 1;
	}
	
	std::string pidfilename = "/var/run/imspector.pid";	
	if (!options["pidfilename"].empty()) pidfilename = options["pidfilename"];
	
	openlog("imspector", debugmode ? LOG_PERROR : 0, LOG_DAEMON);

	if (!debugmode)
	{
		if (fork() == 0)
		{
			/* We are the child. */
			int nullfd;
			if ((nullfd = open("/dev/null", O_WRONLY, 0)) < 0)
			{
				syslog(LOG_ERR, "Error: Couldn't open /dev/null");
				return 1;
			}
			dup2(nullfd, STDIN_FILENO);
			dup2(nullfd, STDOUT_FILENO);
			dup2(nullfd, STDERR_FILENO);
			close(nullfd);
			setsid();
		}
		else
		{
			/* We are the parent so can disapear now. */
			return 0;
		}
	}

	/* Change the file mode mask. */
	umask(0002); 
	
	/* Check its running and write the pidfile. */
	if (checkforrunning(pidfilename))
	{
		syslog(LOG_ERR, "Error: IMSpector is already running");
		return 1;
	}
	if (!writepidfile(pidfilename))
	{
		syslog(LOG_ERR, "Error: Couldn't write PID file");
		return 1;
	}
		
	/* Change the current working directory */
	if ((chdir("/")) < 0)
	{
		syslog(LOG_ERR, "Error: Couldn't change dir to /");
		return 1;
	}
	
	/* Drop privs. */
	if (!options["group"].empty())
	{
		int gid = getgidfromname(options["group"]);
		if (gid < 0)
		{
			syslog(LOG_ERR, "Error: Couldn't lookup group %s\n", options["group"].c_str());
			return 1;
		}
		if (setgid((gid_t) gid) < 0)
		{
			syslog(LOG_ERR, "Error: Couldn't change to group %s (%d)\n", options["group"].c_str(), gid);
			return 1;
		}
	}
	if (!options["user"].empty())
	{
		int uid = getuidfromname(options["user"]);
		if (uid < 0)
		{
			syslog(LOG_ERR, "Error: Couldn't lookup user %s\n", options["user"].c_str());
			return 1;
		}
		if (setuid((uid_t) uid) < 0)
		{
			syslog(LOG_ERR, "Error: Couldn't change to user %s (%d)\n", options["user"].c_str(), uid);
			return 1;
		}
	}	

	/* Prepare SSL support. */
	class SSLState sslstate;
	
	if (options["ssl"] == "on")
#ifdef HAVE_SSL
		if (!sslstate.init(options, debugmode)) return 1;
#else
	{
		syslog(LOG_ERR, "Error: SSL support is not compiled in");
		return 1;
	}
#endif
	
	std::string port = "16667";
	if (!options["port"].empty()) port = options["port"];
	std::string httpport = "";
	if (!options["http_port"].empty()) httpport = options["http_port"];

	if (port.empty() && httpport.empty())
	{
		syslog(LOG_ERR, "Error: No ports configured; not starting");
		return 1;
	}
	
	std::string listenaddr = "0.0.0.0";
	if (!options["listenaddr"].empty()) listenaddr = options["listenaddr"];

#if not defined (SO_BINDTODEVICE)
	if (!options["interface"].empty())
		syslog(LOG_INFO, "Binding to interface is not supported on this platform");
#endif

	int globret = 0;

	/* Glob the plugins by a simple wildcard. */
	glob_t protocolpluginsglob;
	
	memset(&protocolpluginsglob, 0, sizeof(glob_t));
	
	std::string protocolpluginmatch = DEFAULT_PLUGIN_DIR;
	if (!options["plugin_dir"].empty()) protocolpluginmatch = options["plugin_dir"];
	protocolpluginmatch += "/*protocolplugin.so";

	globret = glob(protocolpluginmatch.c_str(), GLOB_DOOFFS, NULL, &protocolpluginsglob);

	if (globret && globret != GLOB_NOMATCH)
	{
		syslog(LOG_ERR, "Error: Couldn't get list of protocol plugins");
		return 1;
	}

	/* Load the plugins and push them onto a vector. */
	for (size_t c = 0; c < (size_t) protocolpluginsglob.gl_pathc; c++)
	{
		/* If enabled load the plugin */
		ProtocolPlugin myprotocolplugin;
		if (!myprotocolplugin.loadplugin(protocolpluginsglob.gl_pathv[c])) { return 1; }
		if (myprotocolplugin.callinitprotocolplugin(options, debugmode))
		{
			syslog(LOG_INFO, "Protocol Plugin name: %s", myprotocolplugin.protocolplugininfo.pluginname.c_str());
			protocolplugins.push_back(myprotocolplugin);
		}
	}

	globfree(&protocolpluginsglob);
	
	/* Glob the filter plugins by a simple wildcard. */
	glob_t filterpluginsglob;
	
	memset(&filterpluginsglob, 0, sizeof(glob_t));
	
	std::string filterpluginmatch = DEFAULT_PLUGIN_DIR;
	if (!options["plugin_dir"].empty()) filterpluginmatch = options["plugin_dir"];
	filterpluginmatch += "/*filterplugin.so";

	globret = glob(filterpluginmatch.c_str(), GLOB_DOOFFS, NULL, &filterpluginsglob);

	if (globret && globret != GLOB_NOMATCH)
	{
		syslog(LOG_ERR, "Error: Couldn't get list of filter plugins");
		return 1;
	}

	/* Load the plugins and push them onto a vector. */
	for (size_t c = 0; c < (size_t) filterpluginsglob.gl_pathc; c++)
	{
		FilterPlugin myfilterplugin;
		if (!myfilterplugin.loadplugin(filterpluginsglob.gl_pathv[c])) { return 1; }
		if (myfilterplugin.callinitfilterplugin(options, debugmode))
		{
			syslog(LOG_INFO, "Filter Plugin name: %s", myfilterplugin.filterplugininfo.pluginname.c_str());
			filterplugins.push_back(myfilterplugin);
		}
	}
	
	globfree(&filterpluginsglob);
	
	/* Glob the responder plugins by a simple wildcard. */
	glob_t responderpluginsglob;
	
	memset(&responderpluginsglob, 0, sizeof(glob_t));
	
	std::string responderpluginmatch = DEFAULT_PLUGIN_DIR;
	if (!options["plugin_dir"].empty()) responderpluginmatch = options["plugin_dir"];
	responderpluginmatch += "/*responderplugin.so";

	globret = glob(responderpluginmatch.c_str(), GLOB_DOOFFS, NULL, &responderpluginsglob);
	
	if (globret && globret != GLOB_NOMATCH)
	{
		syslog(LOG_ERR, "Error: Couldn't get list of responder plugins");
		return 1;
	}

	/* Load the plugins and push them onto a vector. */
	for (size_t c = 0; c < (size_t) responderpluginsglob.gl_pathc; c++)
	{
		ResponderPlugin myresponderplugin;
		if (!myresponderplugin.loadplugin(responderpluginsglob.gl_pathv[c])) { return 1; }
		if (myresponderplugin.callinitresponderplugin(options, debugmode))
		{
			syslog(LOG_INFO, "Responder Plugin name: %s", myresponderplugin.responderplugininfo.pluginname.c_str());
			responderplugins.push_back(myresponderplugin);
		}
	}
	
	globfree(&responderpluginsglob);

	/* Create the logging socket. */		
	class Socket loggingsock(AF_UNIX, SOCK_STREAM);

	if (!loggingsock.listensocket(LOGGING_SOCKET))
	{
		syslog(LOG_ERR, "Error: Couldn't bind to logging socket");
		return false;
	}

	/* Fork off the logging process. */
	switch (fork())
	{
		/* An error occured. */
		case -1:
			syslog(LOG_ERR, "Error: Fork failed: %s", strerror(errno));
			return 1;
			
		/* In the child. */
		case 0:
			loggingprocess(options, debugmode, loggingsock);
			debugprint(debugmode, "Finished the logging process");
			exit(0);
			
		/* In the parent. */
		default:
			break;
	}		

	/* Setup our signal handlers. */
	struct sigaction sa;
	memset(&sa, 0, sizeof(struct sigaction));

	sa.sa_handler = fireman;
	if (sigaction(SIGCHLD, &sa, NULL))
	{
		syslog(LOG_ERR, "Error: Unable to set SIGCHLD handler");
		return 1;
	}
	
	/* Setup handler for SIGTERM. */
	sa.sa_handler = sigterm;
	if (sigaction(SIGTERM, &sa, NULL))
	{
		syslog(LOG_ERR, "Error: Unable to set SIGTERM handler");
		return 1;
	}
	
	/* Handle ctrl+c for debugging. Use the SIGTERM handler. */
	sa.sa_handler = sigterm;
	if (sigaction(SIGINT, &sa, NULL))
	{
		syslog(LOG_ERR, "Error: Unable to set SIGTERM handler");
		return 1;
	}

	/* Ignore SIGPIPE. We will treat those as in-sequrence errors. */
	sa.sa_handler = SIG_IGN;
	if (sigaction(SIGPIPE, &sa, NULL))
	{
		syslog(LOG_ERR, "Error: Unable to ignore SIGPIPE");
		return 1;
	}
	
	class Socket serversock(AF_INET, SOCK_STREAM);
	class Socket httpserversock(AF_INET, SOCK_STREAM);
	
	if (!port.empty())
	{
		if (!serversock.listensocket(listenaddr + ":" + port))
		{
			syslog(LOG_ERR, "Error: Couldn't bind to non-HTTP socket.");
			return 1;
		}

		debugprint(debugmode, "Non-HTTP port listening on %s:%s", listenaddr.c_str(),port.c_str());
	}
	
	if (!httpport.empty())
	{
		if (!httpserversock.listensocket(listenaddr + ":" + httpport))
		{
			syslog(LOG_ERR, "Error: Couldn't bind to HTTP socket.");
			return 1;
		}

		debugprint(debugmode, "HTTP port listening on %s:%s", listenaddr.c_str(),
			httpport.c_str());
	}
	
	while (!quitting)
	{
	 	fd_set rfds;
		struct timeval tv;
	
		FD_ZERO(&rfds);
		if (!port.empty()) FD_SET(serversock.getfd(), &rfds);
		if (!httpport.empty()) FD_SET(httpserversock.getfd(), &rfds);

		tv.tv_sec = 300;
		tv.tv_usec = 0;
		
		int result = select(FD_SETSIZE, &rfds, NULL, NULL, &tv);
		if (result == -1) continue;

		if (!port.empty())
		{
			if (FD_ISSET(serversock.getfd(), &rfds))
				handleserversock(options, debugmode, sslstate, serversock, false);
		}		
		if (!httpport.empty())
		{
			if (FD_ISSET(httpserversock.getfd(), &rfds))
				handleserversock(options, debugmode, sslstate, httpserversock, true);
		}
	}
	
	serversock.closesocket();
	httpserversock.closesocket();
	loggingsock.closesocket();
	
#ifdef HAVE_SSL
	sslstate.free();
#endif
	
	/* Ignore SIGTERM so we can send it to ourselves without running the signal
	 * handler again. */
	sa.sa_handler = SIG_IGN;
	if (sigaction(SIGTERM, &sa, NULL))
	{
		syslog(LOG_ERR, "Error: Unable to ignore SIGTERM");
		return 1;
	}
	
	/* Handle ctrl+c for debugging. */
	sa.sa_handler = SIG_IGN;
	if (sigaction(SIGINT, &sa, NULL))
	{
		syslog(LOG_ERR, "Error: Unable to ignore SIGTERM");
		return 1;
	}	
	kill(0, SIGTERM);
	
	syslog(LOG_INFO, "Good-bye");
	closelog();
	
	return 0;
}

/* This is basically a wrapper on the fork call. It is needed because handling
 * non-HTTP sockets and handling HTTP sockets is very simular. */
bool handleserversock(class Options &options, bool debugmode,
	class SSLState &sslstate, class Socket &serversock, bool http)
{
	std::string clientaddress;
	class Socket clientsock(AF_INET, SOCK_STREAM);

	if (!serversock.awaitconnection(clientsock, clientaddress)) return false;

	debugprint(debugmode, "%s connection from: %s\n", http ? "HTTP" : "Non-HTTP", 
		clientaddress.c_str());

	switch (fork())
	{
		/* An error occured. */
		case -1:
			syslog(LOG_ERR, "Error: Fork failed: %s", strerror(errno));
			clientsock.closesocket();
			serversock.closesocket();
			return 1;
		
		/* In the child. */
		case 0:
			doproxy(options, debugmode, clientsock, clientaddress, sslstate, http);
			clientsock.closesocket();
			debugprint(debugmode, "Finished with child: %s", clientaddress.c_str());
			exit(0);
			
		/* In the parent. */
		default:
			clientsock.closesocket();
			break;
	}

	return true;
}

bool doproxy(class Options &options, bool debugmode, class Socket &clientsock, std::string &clientaddress,
	class SSLState &sslstate, bool http)
{
	if (!clearsighandlers()) return false;

	std::string responseprefix = options["response_prefix"];
	if (responseprefix.empty()) responseprefix = DEFAULT_RESPONSE_PREFIX;

	std::string responsepostfix = options["response_postfix"];
	if (responsepostfix.empty()) responsepostfix = DEFAULT_RESPONSE_POSTFIX;	

	std::string redirectaddress;

	if (!http)
	{
		/* If this a non HTTP connection (ie a redirect), then get the original
		 * address from the socket layer. */
		redirectaddress = clientsock.getredirectaddress();
	}
	else
	{
		/* Otherwise we need to do some initial proxy trickery. */
		char string[STRING_SIZE];

		/* Get the destination address.  This will be in the form of:
		 * CONNECT host:port HTTP/1.0 */
		memset(string, 0, STRING_SIZE);
		if ((clientsock.recvline(string, STRING_SIZE)) < 0) return false;
		stripnewline(string);
		debugprint(debugmode, "HTTP proxy received: %s", string);
		
		std::string command;
		std::vector<std::string> args; int argc;

		char *s;
		s = chopline(string, command, args, argc);
		
		/* Pull down the rest of the clients headers. This might include
		 * authentication info, which will do something with some day. */
		do
		{
			memset(string, 0, STRING_SIZE);
			if ((clientsock.recvline(string, STRING_SIZE)) < 0) return false;
			stripnewline(string);
			debugprint(debugmode, "HTTP proxy received: %s", string);
		}
		while (strlen(string));
		
		/* Handle unknown commands. */
		if (command != "CONNECT" || !argc)
		{
			sendhttpresponse(http, clientsock, "400 Bad request",
				stringprintf("Bad request: %s", command.c_str()));

			return false;
		}
		
		redirectaddress = args[0];
	}
	
	debugprint(debugmode, "Client is connecting to: %s", redirectaddress.c_str());
	
	const char *colon = strchr(redirectaddress.c_str(), ':');
	uint16_t redirectport = 0;
	if (colon) redirectport = htons((uint16_t) atol(colon + 1));
	
	class ProtocolPlugin *pprotocolplugin = NULL;
#ifdef HAVE_SSL
	bool sslport = false;
#endif
	/* Now clever bit: work out what plugin to use. */
	for (std::vector<class ProtocolPlugin>::iterator i = protocolplugins.begin();
		i != protocolplugins.end(); i++)
	{
		if ((*i).protocolplugininfo.port && (*i).protocolplugininfo.port == redirectport)
		{
			pprotocolplugin = &(*i);
			break;
		}
#ifdef HAVE_SSL
		if ((*i).protocolplugininfo.sslport && (*i).protocolplugininfo.sslport == redirectport)
		{
			sslport = true;
			pprotocolplugin = &(*i);
			break;
		}
#endif
	}
	
	/* Dunno what to do with that thing. Caller will close for us. */
	if (!pprotocolplugin) 
	{
		sendhttpresponse(http, clientsock, "400 Bad request",
			stringprintf("Not permitted, connection to port: %d", ntohs(redirectport)));

		syslog(LOG_ERR, "Error: Don't know how to handle connection to %s\n", redirectaddress.c_str());
		return false;
	}

	/* Complete the connection. */
	std::string interface = "";
	if (!options["interface"].empty()) interface = options["interface"];

	class Socket imserversock(AF_INET, SOCK_STREAM);
	 
	if (!(connectsockethttperror(imserversock, redirectaddress, interface, clientsock, http)))
		return false;

	/* Send the "we have connected" message back to the client, if in HTTP
	 * mode. */
	if (!sendhttpresponse(http, clientsock, "200 Connection established", ""))
		return false;

#ifdef HAVE_SSL
	if (sslport)
	{
		debugprint(debugmode, "SSL port, starting SSL on both sides");
		
		/* Attempt SSL to the IM server. */
		if (sslstate.imserversocktossl(imserversock))
		{
			/* If the IM server can go SSL, then handshake with the client. */
			if (!sslstate.clientsocktossl(clientsock)) return false;
		}
		else
		{
			/* Otherwise, if the SSL connection to the IM server failed, then 
			 * close the socket and reopen it. No data has been exchanged with
			 * the IM client at this point, so that is "untained" by the
			 * failure. */
			debugprint(debugmode, "SSL to IM server failed, switching back to clear-text");

			imserversock.closesocket();
			
			if (!(connectsockethttperror(imserversock, redirectaddress, interface, clientsock, http)))
				return false;
		}
	}
#endif

 	fd_set rfds;
	struct timeval tv;
	
	while (true)
	{
		FD_ZERO(&rfds);
		FD_SET(clientsock.getfd(), &rfds);
		FD_SET(imserversock.getfd(), &rfds);

		tv.tv_sec = 300;
		tv.tv_usec = 0;
		
		int result = select(FD_SETSIZE, &rfds, NULL, NULL, &tv);
		if (result == -1) break;
		
		std::vector<struct imevent> imevents;
		char replybuffer[BUFFER_SIZE]; int replylength;
		bool filtered = false;
		
		memset(replybuffer, 0, BUFFER_SIZE);
		
		int retclient = 0; int retimserver = 0;
		
		if (FD_ISSET(clientsock.getfd(), &rfds))
		{
			if ((retclient = pprotocolplugin->callprocesspacket(true, clientsock, replybuffer,
				&replylength, imevents, clientaddress)) > 0) break;
			if (imevents.size() && filterplugins.size())
				filtered = runfilterplugins(replybuffer, imevents);
			if (!filtered)
				if (!(imserversock.sendalldata(replybuffer, replylength))) break;
		}
		
		if (FD_ISSET(imserversock.getfd(), &rfds))
		{
			if ((retimserver = pprotocolplugin->callprocesspacket(false, imserversock, replybuffer,
				&replylength, imevents, clientaddress)) > 0) break;
			if (imevents.size() && filterplugins.size())
				filtered = runfilterplugins(replybuffer, imevents);
			if (!filtered)
				if (!(clientsock.sendalldata(replybuffer, replylength))) break;
		}
		
		if (retclient == -1 || retimserver == -1)
		{
#ifdef HAVE_SSL
			debugprint(debugmode, "Protocol handler wants SSL, starting SSL on both sides");
			if (!sslstate.imserversocktossl(imserversock)) return false;
			if (!sslstate.clientsocktossl(clientsock)) return false;
#else
			syslog(LOG_ERR, "Error: Protocol handler wants to switch to SSL but SSL not compiled in");
#endif
		}
		
		std::vector<struct response> responses;
		
		/* Loop calling all the responder plugins. */
		if (imevents.size())
		{
			for (std::vector<class ResponderPlugin>::iterator i = responderplugins.begin();
				i != responderplugins.end(); i++)
			{
				int rc;
				if ((rc = (*i).callgenerateresponses(imevents, responses)))
					syslog(LOG_ERR, "Error: Unable to generate responses (%d)", rc);
			}
		}
		
		/* Loop generating message packets for responses. */
		for (std::vector<struct response>::iterator i = responses.begin(); i != responses.end(); i++)
		{			
			(*i).text = responseprefix + (*i).text + responsepostfix;
		
			memset(replybuffer, 0, BUFFER_SIZE);
		
			/* May fail because the protocol plugin dosn't yet support generating
			 * messages packets. */
			if (pprotocolplugin->callgeneratemessagepacket((*i), replybuffer, &replylength) > 0)
				continue;
	
			if ((*i).outgoing)
			{
				if (!(imserversock.sendalldata(replybuffer, replylength))) break;
			}
			else
			{
				if (!(clientsock.sendalldata(replybuffer, replylength))) break;
			}
		}

		/* Remove typing events, if told to do so. Also obscure eventdata. */
		if (imevents.size())
			massageimevents(options, imevents);
	
		/* Send imevents vector to logging process. */
		if (imevents.size())
		{
			if (!logimevents(imevents))
				syslog(LOG_ERR, "Error: Unable to communicate with logging process");
		}
	}
	
	imserversock.closesocket();
	
	return true;
}

/* Connect to the im server, and send a message back to the client if HTTP is
 * in use. */
bool connectsockethttperror(class Socket &imserversock, std::string redirectaddress,
	std::string interface, class Socket &clientsock, bool http)
{
	if (!(imserversock.connectsocket(redirectaddress, interface)))
	{
		if (http)
		{
			if (!sendhttpresponse(http, clientsock, "400 Bad request", 
				stringprintf("Unable to connect to: %s", redirectaddress.c_str())))
			{
				syslog(LOG_ERR, "Error: Couldn't send HTTP response to client");
			}
		}
		return false;
	}

	return true;
}

/* Function for sending a HTTP response. response is the errorcode, usermessage will
 * be used to produce an optional HTML page. */
bool sendhttpresponse(bool http, class Socket &sock, std::string response, std::string usermessage)
{
	/* If we are not in HTTP mode, then succeed. */
	if (!http) return true;

	std::string htmlpage;
	
	if (!usermessage.empty())
		htmlpage = stringprintf(
			"<html>" \
			"<head><title>%s</title>" \
			"<body>" \
			"<h1>%s</h1>" \
			"</body>" \
			"</html>",
			usermessage.c_str(), usermessage.c_str());

	std::string httppage = stringprintf(
		"HTTP/1.0 %s\r\n" \
		"Server: IMSpector\r\n" \
		"\r\n" \
		"%s",
		response.c_str(),
		htmlpage.c_str());
		
	if (!(sock.sendalldata(httppage.c_str(), httppage.length())))
	{
		syslog(LOG_ERR, "Error: Couldn't send HTTP response\n");
		return false;
	}
	
	return true;
}

bool loggingprocess(class Options &options, bool debugmode, class Socket &loggingsock)
{
	if (!clearsighandlers()) return false;

	std::vector<class LoggingPlugin> loggingplugins;

	/* Glob the plugins by a simple wildcard. */
	glob_t loggingpluginsglob;
	
	memset(&loggingpluginsglob, 0, sizeof(glob_t));
	
	std::string pluginmatch = DEFAULT_PLUGIN_DIR;
	if (!options["plugin_dir"].empty()) pluginmatch = options["plugin_dir"];
	pluginmatch += "/*loggingplugin.so";

	if (glob(pluginmatch.c_str(), GLOB_DOOFFS, NULL, &loggingpluginsglob))
	{
		syslog(LOG_ERR, "Error: Couldn't get list of logging plugins");
		return false;
	}

	for (size_t c = 0; c < (size_t) loggingpluginsglob.gl_pathc; c++)
	{
		LoggingPlugin myloggingplugin;
		if (!myloggingplugin.loadplugin(loggingpluginsglob.gl_pathv[c])) { return false; }
		if (myloggingplugin.callinitloggingplugin(options, debugmode))
		{
			syslog(LOG_INFO, "Logging Plugin name: %s", myloggingplugin.loggingplugininfo.pluginname.c_str());
			loggingplugins.push_back(myloggingplugin);
		}
	}

	while (true)
	{
		std::vector<struct imevent> imevents;	
		std::string clientaddress;
		class Socket clientsock(AF_UNIX, SOCK_STREAM);
		
		if (!loggingsock.awaitconnection(clientsock, clientaddress)) continue;
		
		char buffer[BUFFER_SIZE];
		memset(buffer, 0, BUFFER_SIZE);
		
		if ((clientsock.recvline(buffer, BUFFER_SIZE)) < 0) continue;
		char *t = strchr(buffer, '\n'); if (t) *t = '\0';	
		
		int imeventcount = atol(buffer);
		
		debugprint(debugmode, "%d elements in imevents", imeventcount);
		
		for (int c = 0; c < imeventcount; c++)
		{
			int eventdatalines = 0; int eventdatalinecounter = 0;
			struct imevent imevent;
			
			for (int line = 0; ; line++)
			{
				if (eventdatalines && eventdatalinecounter >= eventdatalines) break;
					
				if ((clientsock.recvline(buffer, BUFFER_SIZE)) < 0) break;
				char *t = strchr(buffer, '\n'); if (t) *t = '\0';
				
				switch (line)
				{
					case 0:
						imevent.timestamp = atol(buffer);
						break;
					
					case 1:
						imevent.clientaddress = buffer;
						break;
					
					case 2:
						imevent.protocolname = buffer;
						break;
					
					case 3:
						imevent.outgoing = atol(buffer) ? true : false;
						break;
					
					case 4:
						imevent.type = atol(buffer);
						break;
						
					case 5:
						imevent.localid = buffer;
						break;
						
					case 6:
						imevent.remoteid = buffer;
						break;
						
					case 7:
						imevent.filtered = atol(buffer) ? true : false;
						break;

					case 8:
						imevent.categories = buffer;
						break;
						
					case 9:
						eventdatalines = atol(buffer);
						break;
					
					/* Otheriwse its one of the "eventdata" lines. */
					default:
						std::string temp = buffer;
						if (eventdatalinecounter) imevent.eventdata += "\n";
						imevent.eventdata += temp;
						eventdatalinecounter++;
						break;
				}
			}
			
			if (eventdatalines) imevents.push_back(imevent);
		}
		
		clientsock.closesocket();
		
		/* Loop calling all the login plugins. */
		if (imevents.size())
		{
			for (std::vector<class LoggingPlugin>::iterator i = loggingplugins.begin();
				i != loggingplugins.end(); i++)
			{
				int rc;
				if ((rc = (*i).calllogevents(imevents)))
					syslog(LOG_ERR, "Error: Unable to log an event (%d) via %s", rc,
						(*i).loggingplugininfo.pluginname.c_str());
			}
		}
	}
	
	return true;
}
		
bool logimevents(std::vector<struct imevent> &imevents)
{
	char buffer[BUFFER_SIZE];
			
	class Socket loggingsock(AF_UNIX, SOCK_STREAM);
	
	/* Complete the connection. */
	if (!(loggingsock.connectsocket(LOGGING_SOCKET, ""))) return false;
	
	memset(buffer, 0, BUFFER_SIZE);
	
	snprintf(buffer, BUFFER_SIZE - 1, "%ld\n", (long)imevents.size());
	
	if (!loggingsock.sendalldata(buffer, strlen(buffer))) return false;

	for (std::vector<struct imevent>::iterator i = imevents.begin();
		i != imevents.end(); i++)
	{
		memset(buffer, 0, BUFFER_SIZE);
		
		/* Count up the lines. */
		int eventdatalines = 1;	
		for (char *p = (char *)(*i).eventdata.c_str(); *p; p++)
		{
			if (*p == '\n') eventdatalines++;
		}
		
		snprintf(buffer, BUFFER_SIZE - 1,
			"%d\n"
			"%s\n%s\n"
			"%d\n%d\n"
			"%s\n%s\n"
			"%d\n"
			"%s\n"
			"%d\n%s\n",
			(int)(*i).timestamp,
			(*i).clientaddress.c_str(), (*i).protocolname.c_str(),
			(*i).outgoing ? 1 : 0, (*i).type,
			(*i).localid.c_str(), (*i).remoteid.c_str(),
			(*i).filtered ? 1 : 0,
			(*i).categories.c_str(),
			eventdatalines, (*i).eventdata.c_str());
		
		if (!loggingsock.sendalldata(buffer, strlen(buffer))) return false;
	}
	
	loggingsock.closesocket();
	
	return true;
}

bool runfilterplugins(char *buffer, std::vector<struct imevent> &imevents)
{
	char modifiablebuffer[BUFFER_SIZE];
	char originalbuffer[BUFFER_SIZE];
	bool filtered = false;
	
	memset(modifiablebuffer, 0, BUFFER_SIZE);
	memset(originalbuffer, 0, BUFFER_SIZE);
	
	/* Loop looking at each of the message extents. */
	for (std::vector<struct imevent>::iterator i = imevents.begin();
		i != imevents.end(); i++)
	{
		/* Take a private copy of buffer. That way the plugins don't have to
		 * care about the lengths. */
		if ((*i).messageextent.length < 0)
			strncpy(modifiablebuffer, buffer + (*i).messageextent.start, BUFFER_SIZE - 1);
		else
			strncpy(modifiablebuffer, buffer + (*i).messageextent.start, (*i).messageextent.length);
		strncpy(originalbuffer, modifiablebuffer, BUFFER_SIZE - 1);
		
		/* Loop calling all the filter plugins. */
		for (std::vector<class FilterPlugin>::iterator j = filterplugins.begin();
			j != filterplugins.end(); j++)
		{
			if ((*j).callfilter(originalbuffer, modifiablebuffer, (*i)))
				filtered = true;
		}
		
		strncpy(buffer + (*i).messageextent.start, modifiablebuffer, strlen(modifiablebuffer));
	}

	/* Go back through the events and mark with the filtered flag. This is needed
	 * because we may do the block based on the second, or third etc event, but
	 * we have a single of packet that must be blocked (or allowed). */
	for (std::vector<struct imevent>::iterator i = imevents.begin();
		i != imevents.end(); i++)
	{
		(*i).filtered = filtered;
	}
	
	return filtered;
}

/* Simple filter that removes typing events, unless log_typing_events is on. It can
 * also obscure eventdata. */
void massageimevents(class Options &options, std::vector<struct imevent> &imevents)
{
	bool logtyping = false; bool hideeventdata = false;

	if (options["log_typing_events"] == "on") logtyping = true;
	if (options["hide_eventdata"] == "on") hideeventdata = true;

	std::vector<struct imevent>::iterator i = imevents.begin();

	while (i != imevents.end())
	{
		if (hideeventdata) (*i).eventdata = "*HIDDEN*";
	
		if ((*i).type == TYPE_TYPING && !logtyping)
			i = imevents.erase(i);
		else
			i++;
	}
}
		
int getgidfromname(std::string name)
{
	struct group *group = getgrnam(name.c_str());
	if (!group) return -1;
	
	return (int) group->gr_gid;
}

int getuidfromname(std::string name)
{
	struct passwd *user = getpwnam(name.c_str());
	if (!user) return -1;
	
	return (int) user->pw_uid;
}

/* Returns true if the pid pointed to by pidfilename is running. */
bool checkforrunning(std::string pidfilename)
{
	bool rc = false;
	FILE *hfile = fopen(pidfilename.c_str(), "r");
	if (!hfile) return rc;
	
	char buffer[STRING_SIZE];
	memset(buffer, 0, STRING_SIZE);
	
	if (fgets(buffer, STRING_SIZE, hfile))
	{
		char *c; if ((c = strchr(buffer, '\n'))) *c = '\0';
		pid_t pid = (pid_t) atol(buffer);
		if (pid > 0)
		{
			if (!kill(pid, 0)) rc = true;
		}
	}
	
	fclose(hfile);
	
	return rc;
}

/* (Re)creates the pid file. */
bool writepidfile(std::string pidfilename)
{
	unlink(pidfilename.c_str());
	
	bool rc = false;
	FILE *hfile = fopen(pidfilename.c_str(), "w");
	if (!hfile) return rc;
	
	fprintf(hfile, "%d\n", getpid());
	
	fclose(hfile);
	
	rc = true;
	
	return rc;
}

bool clearsighandlers(void)
{
	struct sigaction sa;
	memset(&sa, 0, sizeof(struct sigaction));
	
	sa.sa_handler = SIG_DFL;
	
	if (sigaction(SIGCHLD, &sa, NULL))
	{
		syslog(LOG_ERR, "Error: Unable to default SIGCHLD");
		return false;
	}
	if (sigaction(SIGTERM, &sa, NULL))
	{
		syslog(LOG_ERR, "Error: Unable to default SIGTERM");
		return false;
	}
	
	return true;
}

/* Fireman collects the falling children. */
void fireman(int signal)
{
	int wstatus;

	while (wait3(&wstatus, WNOHANG, NULL) > 0);
}

/* SIGTERM handler simply sets the quitting variable so the mainlop ends. */
void sigterm(int signal)
{
	quitting = true;
}
