/*
 *  $Id: crop.c 25178 2022-12-16 13:24:16Z yeti-dn $
 *  Copyright (C) 2003-2022 David Necas (Yeti), Petr Klapetek.
 *  E-mail: yeti@gwyddion.net, klapetek@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <string.h>
#include <gtk/gtk.h>
#include <libgwyddion/gwymacros.h>
#include <libgwyddion/gwymath.h>
#include <libgwymodule/gwymodule-tool.h>
#include <libprocess/datafield.h>
#include <libgwydgets/gwystock.h>
#include <app/gwyapp.h>

enum {
    PARAM_KEEP_OFFSETS,
    PARAM_NEW_CHANNEL,
    PARAM_HOLD_SELECTION,
};

#define GWY_TYPE_TOOL_CROP            (gwy_tool_crop_get_type())
#define GWY_TOOL_CROP(obj)            (G_TYPE_CHECK_INSTANCE_CAST((obj), GWY_TYPE_TOOL_CROP, GwyToolCrop))
#define GWY_IS_TOOL_CROP(obj)         (G_TYPE_CHECK_INSTANCE_TYPE((obj), GWY_TYPE_TOOL_CROP))
#define GWY_TOOL_CROP_GET_CLASS(obj)  (G_TYPE_INSTANCE_GET_CLASS((obj), GWY_TYPE_TOOL_CROP, GwyToolCropClass))

typedef struct _GwyToolCrop      GwyToolCrop;
typedef struct _GwyToolCropClass GwyToolCropClass;

struct _GwyToolCrop {
    GwyPlainTool parent_instance;

    GwyParams *params;

    GwyRectSelectionLabels *rlabels;
    GwyParamTable *table;
    gdouble rsel[4];
    gint isel[4];

    /* potential class data */
    GType layer_type_rect;
};

struct _GwyToolCropClass {
    GwyPlainToolClass parent_class;
};

static gboolean     module_register                (void);
static GwyParamDef* define_module_params           (void);
static GType        gwy_tool_crop_get_type         (void)                      G_GNUC_CONST;
static void         gwy_tool_crop_finalize         (GObject *object);
static void         gwy_tool_crop_init_dialog      (GwyToolCrop *tool);
static void         gwy_tool_crop_data_switched    (GwyTool *gwytool,
                                                    GwyDataView *data_view);
static void         gwy_tool_crop_data_changed     (GwyPlainTool *plain_tool);
static void         gwy_tool_crop_response         (GwyTool *tool,
                                                    gint response_id);
static void         gwy_tool_crop_selection_changed(GwyPlainTool *plain_tool,
                                                    gint hint);
static void         gwy_tool_crop_apply            (GwyToolCrop *tool);
static void         update_selected_rectangle      (GwyToolCrop *tool);

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Crop tool, crops data to smaller size."),
    "Yeti <yeti@gwyddion.net>",
    "3.1",
    "David Nečas (Yeti) & Petr Klapetek",
    "2003",
};

GWY_MODULE_QUERY2(module_info, crop)

G_DEFINE_TYPE(GwyToolCrop, gwy_tool_crop, GWY_TYPE_PLAIN_TOOL)

static gboolean
module_register(void)
{
    gwy_tool_func_register(GWY_TYPE_TOOL_CROP);

    return TRUE;
}

static GwyParamDef*
define_module_params(void)
{
    static GwyParamDef *paramdef = NULL;

    if (paramdef)
        return paramdef;

    paramdef = gwy_param_def_new();
    gwy_param_def_set_function_name(paramdef, "crop");
    gwy_param_def_add_boolean(paramdef, PARAM_KEEP_OFFSETS, "keep_offsets", _("Keep lateral offsets"), FALSE);
    gwy_param_def_add_boolean(paramdef, PARAM_NEW_CHANNEL, "new_channel", _("Create new image"), TRUE);
    gwy_param_def_add_hold_selection(paramdef, PARAM_HOLD_SELECTION, "hold_selection", NULL);

    return paramdef;
}

static void
gwy_tool_crop_class_init(GwyToolCropClass *klass)
{
    GwyPlainToolClass *ptool_class = GWY_PLAIN_TOOL_CLASS(klass);
    GwyToolClass *tool_class = GWY_TOOL_CLASS(klass);
    GObjectClass *gobject_class = G_OBJECT_CLASS(klass);

    gobject_class->finalize = gwy_tool_crop_finalize;

    tool_class->stock_id = GWY_STOCK_CROP;
    tool_class->title = _("Crop");
    tool_class->tooltip = _("Crop data");
    tool_class->prefix = "/module/crop";
    tool_class->data_switched = gwy_tool_crop_data_switched;
    tool_class->response = gwy_tool_crop_response;

    ptool_class->data_changed = gwy_tool_crop_data_changed;
    ptool_class->selection_changed = gwy_tool_crop_selection_changed;
}

static void
gwy_tool_crop_finalize(GObject *object)
{
    GwyToolCrop *tool = GWY_TOOL_CROP(object);

    gwy_params_save_to_settings(tool->params);
    GWY_OBJECT_UNREF(tool->params);

    G_OBJECT_CLASS(gwy_tool_crop_parent_class)->finalize(object);
}

static void
gwy_tool_crop_init(GwyToolCrop *tool)
{
    GwyPlainTool *plain_tool = GWY_PLAIN_TOOL(tool);

    tool->layer_type_rect = gwy_plain_tool_check_layer_type(plain_tool, "GwyLayerRectangle");
    if (!tool->layer_type_rect)
        return;

    tool->params = gwy_params_new_from_settings(define_module_params());

    plain_tool->lazy_updates = TRUE;
    gwy_plain_tool_connect_selection(plain_tool, tool->layer_type_rect, "rectangle");
    gwy_plain_tool_enable_selection_holding(plain_tool);

    gwy_tool_crop_init_dialog(tool);
}

static void
gwy_tool_crop_rect_updated(GwyToolCrop *tool)
{
    GwyPlainTool *plain_tool = GWY_PLAIN_TOOL(tool);

    gwy_rect_selection_labels_select(tool->rlabels, plain_tool->selection, plain_tool->data_field);
}

static void
gwy_tool_crop_init_dialog(GwyToolCrop *tool)
{
    GtkDialog *dialog = GTK_DIALOG(GWY_TOOL(tool)->dialog);
    GwyParamTable *table;

    tool->rlabels = gwy_rect_selection_labels_new(TRUE, G_CALLBACK(gwy_tool_crop_rect_updated), tool);
    gtk_box_pack_start(GTK_BOX(dialog->vbox), gwy_rect_selection_labels_get_table(tool->rlabels), FALSE, FALSE, 0);

    table = tool->table = gwy_param_table_new(tool->params);
    gwy_param_table_append_checkbox(table, PARAM_KEEP_OFFSETS);
    gwy_param_table_append_checkbox(table, PARAM_NEW_CHANNEL);
    gwy_param_table_append_hold_selection(table, PARAM_HOLD_SELECTION);
    gwy_plain_tool_add_param_table(GWY_PLAIN_TOOL(tool), table);
    gtk_box_pack_start(GTK_BOX(dialog->vbox), gwy_param_table_widget(table), FALSE, FALSE, 0);

    gwy_plain_tool_add_clear_button(GWY_PLAIN_TOOL(tool));
    gwy_tool_add_hide_button(GWY_TOOL(tool), FALSE);
    gtk_dialog_add_button(dialog, GTK_STOCK_APPLY, GTK_RESPONSE_APPLY);
    gtk_dialog_set_default_response(dialog, GTK_RESPONSE_APPLY);
    gtk_dialog_set_response_sensitive(dialog, GTK_RESPONSE_APPLY, FALSE);
    gwy_help_add_to_tool_dialog(dialog, GWY_TOOL(tool), GWY_HELP_NO_BUTTON);

    gtk_widget_show_all(dialog->vbox);
}

static void
gwy_tool_crop_data_switched(GwyTool *gwytool,
                            GwyDataView *data_view)
{
    GwyPlainTool *plain_tool = GWY_PLAIN_TOOL(gwytool);
    GwyToolCrop *tool = GWY_TOOL_CROP(gwytool);
    gboolean ignore = (data_view == plain_tool->data_view);

    GWY_TOOL_CLASS(gwy_tool_crop_parent_class)->data_switched(gwytool, data_view);

    if (ignore || plain_tool->init_failed)
        return;

    if (data_view) {
        gwy_object_set_or_reset(plain_tool->layer, tool->layer_type_rect,
                                "is-crop", TRUE,
                                "editable", TRUE,
                                "focus", -1,
                                NULL);
        gwy_selection_set_max_objects(plain_tool->selection, 1);
        gwy_plain_tool_hold_selection(plain_tool, gwy_params_get_flags(tool->params, PARAM_HOLD_SELECTION));
    }
}

static void
gwy_tool_crop_data_changed(GwyPlainTool *plain_tool)
{
    update_selected_rectangle(GWY_TOOL_CROP(plain_tool));
}

static void
gwy_tool_crop_response(GwyTool *tool,
                       gint response_id)
{
    GWY_TOOL_CLASS(gwy_tool_crop_parent_class)->response(tool, response_id);

    if (response_id == GTK_RESPONSE_APPLY)
        gwy_tool_crop_apply(GWY_TOOL_CROP(tool));
}

static void
gwy_tool_crop_selection_changed(GwyPlainTool *plain_tool,
                                gint hint)
{
    g_return_if_fail(hint <= 0);
    update_selected_rectangle(GWY_TOOL_CROP(plain_tool));
}

static void
crop_one_field(GwyDataField *field,
               const gint *isel,
               const gdouble *sel,
               gboolean keep_offsets)
{
    gdouble xoff = (keep_offsets ? gwy_data_field_get_xoffset(field) + sel[0] : 0.0);
    gdouble yoff = (keep_offsets ? gwy_data_field_get_yoffset(field) + sel[1] : 0.0);

    gwy_data_field_resize(field, isel[0], isel[1], isel[2]+1, isel[3]+1);
    gwy_data_field_set_xoffset(field, xoff);
    gwy_data_field_set_yoffset(field, yoff);
}

static void
gwy_tool_crop_apply(GwyToolCrop *tool)
{
    GwyPlainTool *plain_tool = GWY_PLAIN_TOOL(tool);
    GwyContainer *container = plain_tool->container;
    GwyDataField *field = plain_tool->data_field, *mask = plain_tool->mask_field, *show = plain_tool->show_field;
    gboolean keep_off = gwy_params_get_boolean(tool->params, PARAM_KEEP_OFFSETS);
    gboolean new_channel = gwy_params_get_boolean(tool->params, PARAM_NEW_CHANNEL);
    gint n, id, oldid = plain_tool->id;
    GQuark quarks[3];
    gint isel[4];
    gdouble rsel[4];

    g_return_if_fail(plain_tool->id >= 0 && field != NULL);

    if (!gwy_selection_get_data(plain_tool->selection, NULL)) {
        g_warning("Apply invoked when no selection is present");
        return;
    }

    gwy_assign(isel, tool->isel, 4);
    gwy_assign(rsel, tool->rsel, 4);

    if (new_channel) {
        field = gwy_data_field_duplicate(field);
        crop_one_field(field, isel, rsel, keep_off);
        id = gwy_app_data_browser_add_data_field(field, container, TRUE);
        g_object_unref(field);
        gwy_app_sync_data_items(container, container, oldid, id, FALSE,
                                GWY_DATA_ITEM_GRADIENT,
                                GWY_DATA_ITEM_RANGE_TYPE,
                                GWY_DATA_ITEM_MASK_COLOR,
                                GWY_DATA_ITEM_REAL_SQUARE,
                                0);
        gwy_app_set_data_field_title(container, id, _("Detail"));
        gwy_params_save_to_settings(tool->params);   /* Ensure correct parameters in the log. */
        gwy_app_channel_log_add(container, oldid, id, "tool::GwyToolCrop", NULL);

        if (mask) {
            mask = gwy_data_field_duplicate(mask);
            crop_one_field(mask, isel, rsel, keep_off);
            gwy_container_set_object(container, gwy_app_get_mask_key_for_id(id), mask);
            g_object_unref(mask);
        }

        if (show) {
            show = gwy_data_field_duplicate(show);
            crop_one_field(show, isel, rsel, keep_off);
            gwy_container_set_object(container, gwy_app_get_show_key_for_id(id), show);
            g_object_unref(show);
        }
    }
    else {
        n = 0;
        quarks[n++] = gwy_app_get_data_key_for_id(oldid);
        if (mask)
            quarks[n++] = gwy_app_get_mask_key_for_id(oldid);
        if (show)
            quarks[n++] = gwy_app_get_show_key_for_id(oldid);
        gwy_app_undo_qcheckpointv(container, n, quarks);

        crop_one_field(field, isel, rsel, keep_off);
        gwy_data_field_data_changed(field);
        if (mask) {
            crop_one_field(mask, isel, rsel, keep_off);
            gwy_data_field_data_changed(mask);
        }
        if (show) {
            crop_one_field(show, isel, rsel, keep_off);
            gwy_data_field_data_changed(show);
        }

        gwy_app_data_clear_selections(container, oldid);
        gwy_plain_tool_log_add(plain_tool);
    }
}

static void
update_selected_rectangle(GwyToolCrop *tool)
{
    GwyPlainTool *plain_tool = GWY_PLAIN_TOOL(tool);
    GwySelection *selection = plain_tool->selection;
    GwyDataField *field = plain_tool->data_field;
    gint n = selection ? gwy_selection_get_data(selection, NULL) : 0;
    gint xres, yres;

    gtk_dialog_set_response_sensitive(GTK_DIALOG(GWY_TOOL(tool)->dialog), GTK_RESPONSE_APPLY, FALSE);
    if (n != 1 || !field) {
        gwy_rect_selection_labels_fill(tool->rlabels, NULL, NULL, tool->rsel, tool->isel);
        return;
    }

    gwy_rect_selection_labels_fill(tool->rlabels, selection, field, tool->rsel, tool->isel);

    /* Make Apply insensitive when the full image is selected, for whatever reason. There is nothting to crop then. */
    xres = gwy_data_field_get_xres(field);
    yres = gwy_data_field_get_yres(field);
    if (tool->isel[2] - tool->isel[0] == xres-1 && tool->isel[3] - tool->isel[1] == yres-1)
        return;

    gtk_dialog_set_response_sensitive(GTK_DIALOG(GWY_TOOL(tool)->dialog), GTK_RESPONSE_APPLY, TRUE);
}

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
