static char rcsid[] = "$Id: d5601f421568d795eb2b24cda81099a97a2dbdbf $";
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#ifndef HAVE_MEMCPY
#define memcpy(d,s,n) bcopy((s),(d),(n))
#endif

#include "transcriptome-search.h"

#include "assert.h"
#include "trpath.h"
#include "trpath-solve.h"

#include "path.h"
#include "path-eval.h"

#include "mem.h"
#include "types.h"
#include "chrnum.h"
#include "reader.h"
#include "oligo.h"

#ifdef LARGE_GENOMES
#include "intersect-large.h"
#endif
#include "intersect.h"		/* For Intersect_approx */
/* #include "intersect-simd.h" */

#ifndef LARGE_GENOMES
#include "merge-diagonals-simd-uint4.h"
#elif !defined(HAVE_AVX512) && !defined(HAVE_AVX2)
#include "merge-diagonals-heap.h" /* For Merge_diagonals_large */
#include "merge-diagonals-simd-uint4.h"
#else
#include "merge-diagonals-simd-uint8.h" /* For Merge_diagonals_large */
#include "merge-diagonals-simd-uint4.h"
#endif

#include "transcript.h"
#include "junction.h"
#include "genomebits_count.h"
#include "genomebits_kmer.h"
#include "spliceends.h"


/* Merging faster than count table */
#define USE_MERGE 1
#define SUBOPT 3


#ifdef DEBUG
#define debug(x) x
#else
#define debug(x)
#endif

/* Processing merged diagonals */
#ifdef DEBUG0
#define debug0(x) x
#else
#define debug0(x)
#endif

/* Exon and path overlap */
#ifdef DEBUG1
#define debug1(x) x
#else
#define debug1(x)
#endif

/* Searching complete */
#ifdef DEBUG2
#define debug2(x) x
#else
#define debug2(x)
#endif

/* Transcriptome alignment details*/
#ifdef DEBUG2A
#define debug2a(x) x
#else
#define debug2a(x)
#endif

/* Searching by ends */
#ifdef DEBUG3
#define debug3(x) x
#else
#define debug3(x)
#endif

/* Disambiguate paths */
#ifdef DEBUG5
#define debug5(x) x
#else
#define debug5(x)
#endif

/* convert_to_path */
#ifdef DEBUG9
#define debug9(x) x
#else
#define debug9(x)
#endif


static int index1interval;
static int index1part_tr;

static Indexdb_T tr_indexdb;

static Transcriptome_T transcriptome;
static EF64_T transcript_ef64;

static Genomebits_T genomebits;
static Genomebits_T genomebits_alt;
static Genomebits_T transcriptomebits;

static int leftreadshift;
static Oligospace_T oligobase_mask;


/* Overwrite values array */
/* Repetitive oligos can calse false indel diagonals */
static int
most_prevalent_uint (int *nloci, UINT4 *values, int nvalues) {
  int max_count, count;
  UINT4 *out, *ptr, *end, *first;

  assert(nvalues > 0);

  ptr = out = &(values[0]);	/* Reset */
  end = &(values[nvalues]);

  max_count = 1;
  while (ptr < end) {
    first = ptr;
    debug0(printf("DIAGONAL %u",*first));
    if (ptr + max_count - 1 >= end) {
      /* End of list fails */
      debug0(printf(" => Goes beyond end of list\n"));
      ptr = end;

    } else if (ptr[max_count - 1] != *first) {
      /* Fails */
      debug0(printf(" => Fails\n"));
      if (ptr[max_count - 1] != ptr[max_count - 2]) {
	/* Can jump immediately */
	ptr += max_count - 1;
      } else {
	/* Advance forward until we see a new value */
	while (ptr < end && *ptr == *first) {
	  ptr++;
	}
      }

    } else {
      /* Contender */
      ptr += max_count;
      while (ptr < end && *ptr == *first) {
	ptr++;
      }
      debug0(printf(" => Count %d, index %d => printing\n",ptr - first,first - values));
      if ((count = ptr - first) > max_count) {
	out = &(values[0]);	/* Reset */
	max_count = count;
      }
      *out++ = *first;
    }
  }

  *nloci = out - &(values[0]);
  return max_count;
}


#define T Trpath_T


/* Expecting transcriptome to not be a large genome */
void
Transcriptome_search_complete (int *found_score, bool *found_transcriptp,

			       List_T *unextended_sense_paths_gplus, List_T *unextended_sense_paths_gminus,
			       List_T *unextended_antisense_paths_gplus, List_T *unextended_antisense_paths_gminus,

			       List_T *sense_paths_gplus, List_T *sense_paths_gminus,
			       List_T *antisense_paths_gplus, List_T *antisense_paths_gminus,

			       Stage1_T stage1, char *queryuc_ptr, int querylength, Indelinfo_T indelinfo, Mergeinfo_uint4_T mergeinfo,
			       Trcoord_T **tplus_stream_array, int *tplus_streamsize_array, int *tplus_diagterm_array,
			       Trcoord_T **tminus_stream_array, int *tminus_streamsize_array, int *tminus_diagterm_array,
			       int *mismatch_positions_alloc, Compress_T query_compress_fwd, Compress_T query_compress_rev,
			       int nmismatches_allowed,
			       Intlistpool_T intlistpool, Uintlistpool_T uintlistpool, Univcoordlistpool_T univcoordlistpool,
			       Listpool_T listpool, Trpathpool_T trpathpool, Pathpool_T pathpool,
			       Vectorpool_T vectorpool, Transcriptpool_T transcriptpool, Hitlistpool_T hitlistpool,
			       Method_T method) {
  Path_T path;

  int max_tplus_count, max_tminus_count;

  Trcoord_T trdiagonal;
  Trcoord_T *tplus_diagonals, *tminus_diagonals;
  int n_tplus_loci, n_tminus_loci, n_tplus_diagonals, n_tminus_diagonals,
    total_ndiagonals_tplus = 0, total_ndiagonals_tminus = 0, i;
  
  Trcoord_T *positions;
  int npositions;
  int tplus_streami = 0, tminus_streami = 0;

#if 0
  Reader_T reader;
  Oligostate_T last_state = INIT;
  Oligospace_T forward = 0, revcomp = 0, forward_oligo, revcomp_oligo;
#endif

  int querypos, query_lastpos;

  Trnum_T trnum;
  int transcript_genestrand;
  Trcoord_T troffset, trhigh;
  Chrnum_T chrnum;


  query_lastpos = querylength - index1part_tr;
  for (querypos = 0; querypos <= query_lastpos; querypos++) {
    if (stage1->validp[querypos] == false) {
      /* Skip */
    } else {
      if (stage1->tr_plus_retrievedp[querypos] == true) {
	positions = stage1->tr_plus_positions[querypos];
	npositions = stage1->tr_plus_npositions[querypos];
      } else {
	npositions = stage1->tr_plus_npositions[querypos] =
	  Indexdb_ptr(&stage1->tr_plus_positions[querypos],tr_indexdb,
		      stage1->forward_oligos[querypos]);
	positions = stage1->tr_plus_positions[querypos];
	stage1->tr_plus_retrievedp[querypos] = true;
      }
      if (npositions > 0) {
	tplus_stream_array[tplus_streami] = positions;
	tplus_streamsize_array[tplus_streami] = npositions;
	tplus_diagterm_array[tplus_streami] = querylength - querypos;
	total_ndiagonals_tplus += npositions;
	tplus_streami++;
      }

      if (stage1->tr_minus_retrievedp[querypos] == true) {
	positions = stage1->tr_minus_positions[querypos];
	npositions = stage1->tr_minus_npositions[querypos];
      } else {
	npositions = stage1->tr_minus_npositions[querypos] =
	  Indexdb_ptr(&stage1->tr_minus_positions[querypos],tr_indexdb,
		      stage1->revcomp_oligos[querypos]);
	positions = stage1->tr_minus_positions[querypos];
	stage1->tr_minus_retrievedp[querypos] = true;
      }
      if (npositions > 0) {
	tminus_stream_array[tminus_streami] = positions;
	tminus_streamsize_array[tminus_streami] = npositions;
	tminus_diagterm_array[tminus_streami] = querypos + index1part_tr;
	total_ndiagonals_tminus += npositions;
	tminus_streami++;
      }
    }
  }


  tplus_diagonals = Merge_diagonals(&n_tplus_diagonals,tplus_stream_array,
				    tplus_streamsize_array,tplus_diagterm_array,/*nstreams*/tplus_streami,
				    mergeinfo);

  if (n_tplus_diagonals == 0) {
    max_tplus_count = 0;
    n_tplus_loci = 0;
  } else {
    max_tplus_count = most_prevalent_uint(&n_tplus_loci,tplus_diagonals,n_tplus_diagonals);
  }

#ifdef DEBUG2
  printf("max_tplus_count: %d\n",max_tplus_count);
  for (i = 0; i < n_tplus_loci; i++) {
    printf("PLUS_LOCUS %u\n",tplus_diagonals[i]);
  }
#endif

  tminus_diagonals = Merge_diagonals(&n_tminus_diagonals,tminus_stream_array,
				     tminus_streamsize_array,tminus_diagterm_array,/*nstreams*/tminus_streami,
				     mergeinfo);

  if (n_tminus_diagonals == 0) {
    max_tminus_count = 0;
    n_tminus_loci = 0;
  } else {
    max_tminus_count = most_prevalent_uint(&n_tminus_loci,tminus_diagonals,n_tminus_diagonals);
  }


#ifdef DEBUG2
  printf("max_tminus_count: %d\n",max_tminus_count);
  for (i = 0; i < n_tminus_loci; i++) {
    printf("MINUS_LOCUS %u\n",tminus_diagonals[i]);
  }
#endif

  /* tplusp == true */
  if (max_tplus_count < (querylength - index1part_tr)/5 &&
      max_tplus_count < max_tminus_count - SUBOPT) {
    /* Not enough kmers match or less than minus */
    debug2(printf("Not enough tplus kmers match (%d)\n",max_tplus_count));

  } else {
    for (i = 0; i < n_tplus_loci; i++) {
      trdiagonal = tplus_diagonals[i];
      if (trdiagonal /*+qstart:0*/ < (Trcoord_T) querylength) {
	/* Skip */
      } else {
	trnum = EF64_trnum(&troffset,&trhigh,transcript_ef64,trdiagonal - querylength,trdiagonal);
	chrnum = Transcriptome_chrnum(&transcript_genestrand,transcriptome,trnum);

	if (transcript_genestrand > 0) {
	  debug(printf("Complete case 1: tplus, geneplus\n"));
	  if ((path = Trpath_solve_from_diagonals(&(*found_score),&(*found_transcriptp),trdiagonal,
						  /*middle_trdiagonal_qstart*/0,/*middle_trdiagonal_qend*/querylength,
						  /*middle_nmismatches*/0,/*qstart_trdiag*/NULL,/*qend_trdiag*/NULL,
						  /*tplusp*/true,querylength,
						  /*query_compress_tr*/query_compress_fwd,
						  query_compress_fwd,query_compress_rev,
						  mismatch_positions_alloc,trnum,troffset,trhigh,
						  chrnum,/*geneplusp*/true,/*want_lowest_coordinate_p*/true,
						  indelinfo,nmismatches_allowed,
						  /*max_insertionlen*/3,/*max_deletionlen*/3,
						  intlistpool,uintlistpool,univcoordlistpool,listpool,
						  trpathpool,pathpool,vectorpool,transcriptpool,
						  method)) != NULL) {
	    if (Path_unextendedp(path,/*endtrim_allowed*/0,/*allow_ambig_p*/true) == true) {
	      assert(path->plusp == true);
	      *unextended_sense_paths_gplus = Hitlist_push(*unextended_sense_paths_gplus,hitlistpool,(void *) path
							   hitlistpool_trace(__FILE__,__LINE__));
	    } else {
	      assert(path->plusp == true);
	      *sense_paths_gplus = Hitlist_push(*sense_paths_gplus,hitlistpool,(void *) path
						hitlistpool_trace(__FILE__,__LINE__));
	    }
	  }

	} else {
	  debug(printf("Complete case 2: tplus, geneminus\n"));
	  if ((path = Trpath_solve_from_diagonals(&(*found_score),&(*found_transcriptp),trdiagonal,
						  /*middle_trdiagonal_qstart*/0,/*middle_trdiagonal_qend*/querylength,
						  /*middle_nmismatches*/0,/*qstart_trdiag*/NULL,/*qend_trdiag*/NULL,
						  /*tplusp*/true,querylength,
						  /*query_compress_tr*/query_compress_fwd,
						  query_compress_fwd,query_compress_rev,
						  mismatch_positions_alloc,trnum,troffset,trhigh,
						  chrnum,/*geneplusp*/false,/*want_lowest_coordinate_p*/false,
						  indelinfo,nmismatches_allowed,
						  /*max_insertionlen*/3,/*max_deletionlen*/3,
						  intlistpool,uintlistpool,univcoordlistpool,listpool,
						  trpathpool,pathpool,vectorpool,transcriptpool,
						  method)) != NULL) {
	    if (Path_unextendedp(path,/*endtrim_allowed*/0,/*allow_ambig_p*/true) == true) {
	      assert(path->plusp == false);
	      *unextended_sense_paths_gminus = Hitlist_push(*unextended_sense_paths_gminus,hitlistpool,(void *) path
							    hitlistpool_trace(__FILE__,__LINE__));
	    } else {
	      assert(path->plusp == false);
	      *sense_paths_gminus = Hitlist_push(*sense_paths_gminus,hitlistpool,(void *) path
						 hitlistpool_trace(__FILE__,__LINE__));
	    }
	  }
	}
      }
    }

    debug2(printf("tplus loci %d\n",n_tplus_loci));
  }
#if !defined(LARGE_GENOMES) || defined(HAVE_AVX512) || defined(HAVE_AVX2)
  FREE_ALIGN(tplus_diagonals);
#else
  FREE(tplus_diagonals);
#endif


  /* tplusp == false */
  if (max_tminus_count < (querylength - index1part_tr)/5 &&
      max_tminus_count < max_tplus_count - SUBOPT) {
    /* Not enough kmers match or less than plus */
    debug2(printf("Not enough tminus kmers match (%d)\n",max_tminus_count));

  } else {
    for (i = 0; i < n_tminus_loci; i++) {
      trdiagonal = tminus_diagonals[i];
      if (trdiagonal /*+qstart:0*/ < (Trcoord_T) querylength) {
	/* Skip */
      } else {
	trnum = EF64_trnum(&troffset,&trhigh,transcript_ef64,trdiagonal - querylength,trdiagonal);
	chrnum = Transcriptome_chrnum(&transcript_genestrand,transcriptome,trnum);

	if (transcript_genestrand > 0) {
	  debug(printf("Complete case 3: tminus, geneplus\n"));
	  if ((path = Trpath_solve_from_diagonals(&(*found_score),&(*found_transcriptp),trdiagonal,
						  /*middle_trdiagonal_qstart*/0,/*middle_trdiagonal_qend*/querylength,
						  /*middle_nmismatches*/0,/*qstart_trdiag*/NULL,/*qend_trdiag*/NULL,
						  /*tplusp*/false,querylength,
						  /*query_compress_tr*/query_compress_rev,
						  query_compress_fwd,query_compress_rev,
						  mismatch_positions_alloc,trnum,troffset,trhigh,
						  chrnum,/*geneplusp*/true,/*want_lowest_coordinate_p*/true,
						  indelinfo,nmismatches_allowed,
						  /*max_insertionlen*/3,/*max_deletionlen*/3,
						  intlistpool,uintlistpool,univcoordlistpool,listpool,
						  trpathpool,pathpool,vectorpool, transcriptpool,
						  method)) != NULL) {
	    if (Path_unextendedp(path,/*endtrim_allowed*/0,/*allow_ambig_p*/true) == true) {
	      assert(path->plusp == false);
	      *unextended_antisense_paths_gminus = Hitlist_push(*unextended_antisense_paths_gminus,hitlistpool,(void *) path
								hitlistpool_trace(__FILE__,__LINE__));
	    } else {
	      assert(path->plusp == false);
	      *antisense_paths_gminus = Hitlist_push(*antisense_paths_gminus,hitlistpool,(void *) path
						     hitlistpool_trace(__FILE__,__LINE__));
	    }
	  }

	} else {
	  debug(printf("Complete case 4: tminus, geneminus\n"));
	  if ((path = Trpath_solve_from_diagonals(&(*found_score),&(*found_transcriptp),trdiagonal,
						  /*middle_trdiagonal_qstart*/0,/*middle_trdiagonal_qend*/querylength,
						  /*middle_nmismatches*/0,/*qstart_trdiag*/NULL,/*qend_trdiag*/NULL,
						  /*tplusp*/false,querylength,
						  /*query_compress_tr*/query_compress_rev,
						  query_compress_fwd,query_compress_rev,
						  mismatch_positions_alloc,trnum,troffset,trhigh,
						  chrnum,/*geneplusp*/false,/*want_lowest_coordinate_p*/false,
						  indelinfo,nmismatches_allowed,
						  /*max_insertionlen*/3,/*max_deletionlen*/3,
						  intlistpool,uintlistpool,univcoordlistpool,listpool,
						  trpathpool,pathpool,vectorpool,transcriptpool,
						  method)) != NULL) {
	    if (Path_unextendedp(path,/*endtrim_allowed*/0,/*allow_ambig_p*/true) == true) {
	      assert(path->plusp == true);
	      *unextended_antisense_paths_gplus = Hitlist_push(*unextended_antisense_paths_gplus,hitlistpool,(void *) path
							       hitlistpool_trace(__FILE__,__LINE__));
	    } else {
	      assert(path->plusp == true);
	      *antisense_paths_gplus = Hitlist_push(*antisense_paths_gplus,hitlistpool,(void *) path
						    hitlistpool_trace(__FILE__,__LINE__));
	    }
	  }
	}
      }
    }

    debug2(printf("tminus loci %d\n",n_tminus_loci));
  }

#if !defined(LARGE_GENOMES) || defined(HAVE_AVX512) || defined(HAVE_AVX2)
  FREE_ALIGN(tminus_diagonals);
#else
  FREE(tminus_diagonals);
#endif

  return;
}


/* Intersect_approx does not seem to reduce speed and can find some
   alignments that Intersect_exact cannot.  However, this was before
   we implemented a SIMD version of Intersect_exact. */
/* Ultrafast with Intersect_approx */
void
Transcriptome_search_ends (int *found_score, bool *found_transcriptp,

			   List_T *sense_paths_gplus, List_T *sense_paths_gminus,
			   List_T *antisense_paths_gplus, List_T *antisense_paths_gminus,
				  
			   Stage1_T stage1,

			   char *queryuc_ptr, int querylength, Indelinfo_T indelinfo,
			   Compress_T query_compress_fwd, Compress_T query_compress_rev,
			   int nmismatches_allowed, int max_insertionlen, int max_deletionlen,
			   Intlistpool_T intlistpool, Uintlistpool_T uintlistpool,
			   Univcoordlistpool_T univcoordlistpool, Listpool_T listpool,
			   Trpathpool_T trpathpool, Pathpool_T pathpool, Vectorpool_T vectorpool,
			   Transcriptpool_T transcriptpool, Hitlistpool_T hitlistpool,
			   Method_T method) {
  Path_T path;
  
  Trcoord_T *tplus_positions_end5 = NULL, *tminus_positions_end5 = NULL,
    *tplus_positions_end3 = NULL, *tminus_positions_end3 = NULL;
  int n_tplus_positions_end5 = 0, n_tminus_positions_end5 = 0,
    n_tplus_positions_end3 = 0, n_tminus_positions_end3 = 0;
  int tplus_diagterm_end5, tminus_diagterm_end5, tplus_diagterm_end3, tminus_diagterm_end3;

  /* Needed for handling Intersect_approx */
  Trcoord_T trdiagonala, trdiagonalb;

  Trcoord_T trdiagonal;
  Trcoord_T *tplus_diagpairs, *tminus_diagpairs;
  int n_tplus_diagpairs, n_tminus_diagpairs, i, k;
  bool tplus_exactp, tminus_exactp;
  int pos5a, pos3a, pos5b, pos3b;
  int nmismatches_a5, nmismatches_a3, nmismatches_b5, nmismatches_b3;

#if 0
  Reader_T reader;
  Oligospace_T forward, revcomp;
#endif

  int querypos, query_lastpos;
  Oligospace_T forward_oligo, revcomp_oligo;

  Trnum_T trnum;
  int transcript_genestrand;
  Trcoord_T troffset, trhigh;
  Chrnum_T chrnum;


#if 0
  *tplus_positions_5 = *tminus_positions_5 = *tplus_positions_3 = *tminus_positions_3 = (Trcoord_T *) NULL;
  *n_tplus_positions_5 = *n_tminus_positions_5 = *n_tplus_positions_3 = *n_tminus_positions_3 = 0;
#endif
			
  debug3(printf("\n\n***Searching transcriptome by ends\n"));
  debug3(printf("%s\n",queryuc_ptr));


  /* Use oligos computed by Stage1_init */
  querypos = 0;
  while (querypos < index1interval && stage1->validp[querypos] == false) {
    debug3(printf("validp at %d is false\n",querypos));
    querypos++;
  }
  if (querypos < index1interval) {
    forward_oligo = stage1->forward_oligos[querypos];
    tplus_diagterm_end5 = querylength - querypos;
    stage1->tr_plus_npositions[querypos] = n_tplus_positions_end5 = 
      Indexdb_ptr(&tplus_positions_end5,tr_indexdb,forward_oligo);
    stage1->tr_plus_positions[querypos] = tplus_positions_end5;
    stage1->tr_plus_retrievedp[querypos] = true;

    revcomp_oligo = stage1->revcomp_oligos[querypos];
    tminus_diagterm_end5 = querypos + index1part_tr;
    stage1->tr_minus_npositions[querypos] = n_tminus_positions_end5 =
      Indexdb_ptr(&tminus_positions_end5,tr_indexdb,revcomp_oligo);
    stage1->tr_minus_positions[querypos] = tminus_positions_end5;
    stage1->tr_minus_retrievedp[querypos] = true;

    debug3(printf("5' end: %s %s: %d plus positions, %d minus positions\n",
		  Oligo_one_nt(forward_oligo,index1part_tr),Oligo_one_nt(revcomp_oligo,index1part_tr),
		  n_tplus_positions_end5,n_tminus_positions_end5));
  }

  /* Use oligos computed by Stage1_fill_all_oligos */
  if ((query_lastpos = querylength - index1part_tr) >= index1interval - 1) {
    querypos = query_lastpos;  /* querylength - index1part_tr */
    while (querypos > query_lastpos - index1interval && stage1->validp[querypos] == false) {
      querypos--;
    }

    if (querypos > query_lastpos - index1interval) {
      forward_oligo = stage1->forward_oligos[querypos];
      tplus_diagterm_end3 = querylength - querypos;
      stage1->tr_plus_npositions[querypos] = n_tplus_positions_end3 =
	Indexdb_ptr(&tplus_positions_end3,tr_indexdb,forward_oligo);
      stage1->tr_plus_positions[querypos] = tplus_positions_end3;
      stage1->tr_plus_retrievedp[querypos] = true;
      
      revcomp_oligo = stage1->revcomp_oligos[querypos];
      tminus_diagterm_end3 = querypos + index1part_tr;
      stage1->tr_minus_npositions[querypos] = n_tminus_positions_end3 =
	Indexdb_ptr(&tminus_positions_end3,tr_indexdb,revcomp_oligo);
      stage1->tr_minus_positions[querypos] = tminus_positions_end3;
      stage1->tr_minus_retrievedp[querypos] = true;
      
      debug3(printf("3' end: %s %s: %d plus positions, %d minus positions\n",
		    Oligo_one_nt(forward_oligo,index1part_tr),Oligo_one_nt(revcomp_oligo,index1part_tr),
		    n_tplus_positions_end3,n_tminus_positions_end3));
    }
  }
	 
  tplus_diagpairs = Intersect_approx(&tplus_exactp,&n_tplus_diagpairs,
				     tplus_positions_end5,n_tplus_positions_end5,tplus_diagterm_end5,
				     tplus_positions_end3,n_tplus_positions_end3,tplus_diagterm_end3,
				     /*maxdistance*/3);
  tminus_diagpairs = Intersect_approx(&tminus_exactp,&n_tminus_diagpairs,
				      tminus_positions_end5,n_tminus_positions_end5,tminus_diagterm_end5,
				      tminus_positions_end3,n_tminus_positions_end3,tminus_diagterm_end3,
				      /*maxdistance*/3);

  debug3(printf("***Ultrafast transcriptome: exactp %d and %d.  %d plus and %d minus diagpairs\n",
		tplus_exactp,tminus_exactp,n_tplus_diagpairs,n_tminus_diagpairs));

  if (n_tplus_diagpairs == 0) {
    FREE(tplus_diagpairs);	/* Occupies memory even if n_tplus_diagpairs == 0 */

  } else if (tplus_exactp == true) {
    for (i = 0, k = 0; i < n_tplus_diagpairs; i++, k += 2) {
      debug3(printf("tplus diagpairs: %u and %u\n",tplus_diagpairs[k],tplus_diagpairs[k+1]));
      if ((trdiagonal = tplus_diagpairs[k]) == tplus_diagpairs[k+1]) {
	if (trdiagonal /*+qtart:0*/ < (Trcoord_T) querylength) {
	  /* Skip */
	} else {
	  trnum = EF64_trnum(&troffset,&trhigh,transcript_ef64,trdiagonal - querylength,trdiagonal);
	  chrnum = Transcriptome_chrnum(&transcript_genestrand,transcriptome,trnum);

	  if (transcript_genestrand > 0) {
	    debug3(printf("Exact case 1: tplus, geneplus\n"));
	    if ((path = Trpath_solve_exact(&(*found_score),&(*found_transcriptp),
					   trdiagonal,/*pos5*/0,/*pos3*/querylength,
					   /*tplusp*/true,querylength,/*query_compress_tr*/query_compress_fwd,
					   query_compress_fwd,query_compress_rev,
					   trnum,troffset,trhigh,chrnum,/*geneplusp*/true,
					   nmismatches_allowed,
					   intlistpool,uintlistpool,univcoordlistpool,listpool,
					   trpathpool,pathpool,vectorpool,transcriptpool,method)) != NULL) {
	      assert(path->plusp == true);
	      *sense_paths_gplus = Hitlist_push(*sense_paths_gplus,hitlistpool,(void *) path
						hitlistpool_trace(__FILE__,__LINE__));
	    }
	  } else {
	    debug3(printf("Exact case 2: tplus, geneminus\n"));
	    if ((path = Trpath_solve_exact(&(*found_score),&(*found_transcriptp),trdiagonal,/*pos5*/0,/*pos3*/querylength,
					   /*tplusp*/true,querylength,/*query_compress_tr*/query_compress_fwd,
					   query_compress_fwd,query_compress_rev,
					   trnum,troffset,trhigh,chrnum,/*geneplusp*/false,
					   nmismatches_allowed,
					   intlistpool,uintlistpool,univcoordlistpool,listpool,
					   trpathpool,pathpool,vectorpool,transcriptpool,method)) != NULL) {
	      assert(path->plusp == false);
	      *sense_paths_gminus = Hitlist_push(*sense_paths_gminus,hitlistpool,(void *) path
						 hitlistpool_trace(__FILE__,__LINE__));
	    }
	  }
	}
      }
    }

    FREE(tplus_diagpairs);	/* Occupies memory even if n_tplus_diagpairs == 0 */

  } else if (tminus_exactp == true) {
    FREE(tplus_diagpairs);	/* Occupies memory even if n_tplus_diagpairs == 0 */

  } else {
    /* Handle approximate diagpairs below */
  }
    

  if (n_tminus_diagpairs == 0) {
    FREE(tminus_diagpairs);	/* Occupies memory even if n_tminus_diagpairs == 0 */

  } else if (tminus_exactp == true) {
    for (i = 0, k = 0; i < n_tminus_diagpairs; i++, k += 2) {
      debug3(printf("tminus diagpairs: %u and %u\n",tminus_diagpairs[k],tminus_diagpairs[k+1]));
      if ((trdiagonal = tminus_diagpairs[k]) == tminus_diagpairs[k+1]) {
	if (trdiagonal /*+qstart:0*/ < (Trcoord_T) querylength) {
	  /* Skip */
	} else {
	  trnum = EF64_trnum(&troffset,&trhigh,transcript_ef64,trdiagonal - querylength,trdiagonal);
	  chrnum = Transcriptome_chrnum(&transcript_genestrand,transcriptome,trnum);

	  if (transcript_genestrand > 0) {
	    debug3(printf("Exact case 3: tminus, geneplus\n"));
	    if ((path = Trpath_solve_exact(&(*found_score),&(*found_transcriptp),
					   trdiagonal,/*pos5*/0,/*pos3*/querylength,
					   /*tplusp*/false,querylength,/*query_compress_tr*/query_compress_rev,
					   query_compress_fwd,query_compress_rev,
					   trnum,troffset,trhigh,chrnum,/*geneplusp*/true,nmismatches_allowed,
					   intlistpool,uintlistpool,univcoordlistpool,listpool,
					   trpathpool,pathpool,vectorpool,transcriptpool,method)) != NULL) {
	      assert(path->plusp == false);
	      *antisense_paths_gminus = Hitlist_push(*antisense_paths_gminus,hitlistpool,(void *) path
						     hitlistpool_trace(__FILE__,__LINE__));
	    }
	  } else {
	    debug3(printf("Exact case 4: tminus, geneminus\n"));
	    if ((path = Trpath_solve_exact(&(*found_score),&(*found_transcriptp),
					   trdiagonal,/*pos5*/0,/*pos3*/querylength,
					   /*tplusp*/false,querylength,/*query_compress_tr*/query_compress_rev,
					   query_compress_fwd,query_compress_rev,
					   trnum,troffset,trhigh,chrnum,/*geneplusp*/false,nmismatches_allowed,
					   intlistpool,uintlistpool,univcoordlistpool,listpool,
					   trpathpool,pathpool,vectorpool,transcriptpool,method)) != NULL) {
	      assert(path->plusp == true);
	      *antisense_paths_gplus = Hitlist_push(*antisense_paths_gplus,hitlistpool,(void *) path
						    hitlistpool_trace(__FILE__,__LINE__));
	    }
	  }
	}
      }
    }
    FREE(tminus_diagpairs);	/* Occupies memory even if n_tminus_diagpairs == 0 */

  } else if (tplus_exactp == true) {
    FREE(tminus_diagpairs);	/* Occupies memory even if n_tminus_diagpairs == 0 */

  } else {
    /* Handle approximate diagpairs below */
  }

  /* Handle approximate diagpairs */
  if (tplus_exactp == false && tminus_exactp == false) {

    /* Case for n_tplus_diagpairs == 0 already handled above */
    if (n_tplus_diagpairs > 0) {
      for (i = 0, k = 0; i < n_tplus_diagpairs; i++, k += 2) {
	debug3(printf("tplus diagpairs: %u and %u\n",tplus_diagpairs[k],tplus_diagpairs[k+1]));
	trdiagonala = tplus_diagpairs[k];
	trdiagonalb = tplus_diagpairs[k+1];

	pos5a = Genomebits_first_kmer_left(&nmismatches_a5,transcriptomebits,query_compress_fwd,
					   /*univdiagonal*/(Univcoord_T) trdiagonala,querylength,
					   /*pos5*/0,/*pos3*/querylength,/*plusp*/true,/*genestrand*/0,
					   /*query_unk_mismatch_local_p*/false,/*kmer*/index1part_tr);
	pos3a = Genomebits_first_kmer_right(&nmismatches_a3,transcriptomebits,query_compress_fwd,
					    /*univdiagonal*/(Univcoord_T) trdiagonala,querylength,
					    /*pos5*/0,/*pos3*/querylength,/*plusp*/true,/*genestrand*/0,
					    /*query_unk_mismatch_local_p*/false,/*kmer*/index1part_tr);

	pos5b = Genomebits_first_kmer_left(&nmismatches_b5,transcriptomebits,query_compress_fwd,
					   /*univdiagonal*/(Univcoord_T) trdiagonalb,querylength,
					    /*pos5*/0,/*pos3*/querylength,/*plusp*/true,/*genestrand*/0,
					    /*query_unk_mismatch_local_p*/false,/*kmer*/index1part_tr);
	pos3b = Genomebits_first_kmer_right(&nmismatches_b3,transcriptomebits,query_compress_fwd,
					    /*univdiagonal*/(Univcoord_T) trdiagonalb,querylength,
					    /*pos5*/0,/*pos3*/querylength,/*plusp*/true,/*genestrand*/0,
					    /*query_unk_mismatch_local_p*/false,/*kmer*/index1part_tr);
	  
	if (trdiagonala /*+qstart:0*/ < (Trcoord_T) querylength) {
	  /* Skip */

	} else if (pos5a == 0 || pos3b == querylength) {
	  /* Setting a first, b second */
	  trnum = EF64_trnum(&troffset,&trhigh,transcript_ef64,trdiagonala - querylength,trdiagonala);
	  chrnum = Transcriptome_chrnum(&transcript_genestrand,transcriptome,trnum);

	  if (transcript_genestrand > 0) {
	    debug3(printf("Approx case 1: tplus, geneplus\n"));
	    if ((path = Trpath_solve_from_ends(&(*found_score),&(*found_transcriptp),
					       trdiagonala,pos5a,pos3a,trdiagonalb,pos5b,pos3b,
					       /*tplusp*/true,querylength,/*query_compress_tr*/query_compress_fwd,
					       query_compress_fwd,query_compress_rev,
					       trnum,troffset,trhigh,chrnum,/*geneplusp*/true,/*want_lowest_coordinate_p*/true,
					       indelinfo,nmismatches_allowed,
					       max_insertionlen,max_deletionlen,
					       intlistpool,uintlistpool,univcoordlistpool,listpool,
					       trpathpool,pathpool,vectorpool,transcriptpool,method)) != NULL) {
	      assert(path->plusp == true);
	      *sense_paths_gplus = Hitlist_push(*sense_paths_gplus,hitlistpool,(void *) path
						hitlistpool_trace(__FILE__,__LINE__));
	    }
	  } else {
	    debug3(printf("Approx case 2: tplus, geneminus\n"));
	    if ((path = Trpath_solve_from_ends(&(*found_score),&(*found_transcriptp),
					       trdiagonala,pos5a,pos3a,trdiagonalb,pos5b,pos3b,
					       /*tplusp*/true,querylength,/*query_compress_tr*/query_compress_fwd,
					       query_compress_fwd,query_compress_rev,
					       trnum,troffset,trhigh,chrnum,/*geneplusp*/false,/*want_lowest_coordinate_p*/false,
					       indelinfo,nmismatches_allowed,
					       max_insertionlen,max_deletionlen,
					       intlistpool,uintlistpool,univcoordlistpool,listpool,
					       trpathpool,pathpool,vectorpool,transcriptpool,method)) != NULL) {
	      assert(path->plusp == false);
	      *sense_paths_gminus = Hitlist_push(*sense_paths_gminus,hitlistpool,(void *) path
						 hitlistpool_trace(__FILE__,__LINE__));
	    }
	  }

	} else if (pos5b == 0 || pos3a == querylength) {
	  /* Setting b first, a second */
	  trnum = EF64_trnum(&troffset,&trhigh,transcript_ef64,trdiagonala - querylength,trdiagonala);
	  chrnum = Transcriptome_chrnum(&transcript_genestrand,transcriptome,trnum);

	  if (transcript_genestrand > 0) {
	    debug3(printf("Approx case 1: tplus, geneplus\n"));
	    if ((path = Trpath_solve_from_ends(&(*found_score),&(*found_transcriptp),
					       trdiagonalb,pos5b,pos3b,trdiagonala,pos5a,pos3a,
					       /*tplusp*/true,querylength,/*query_compress_tr*/query_compress_fwd,
					       query_compress_fwd,query_compress_rev,
					       trnum,troffset,trhigh,chrnum,/*geneplusp*/true,/*want_lowest_coordinate_p*/true,
					       indelinfo,nmismatches_allowed,
					       max_insertionlen,max_deletionlen,
					       intlistpool,uintlistpool,univcoordlistpool,listpool,
					       trpathpool,pathpool,vectorpool,transcriptpool,method)) != NULL) {
	      assert(path->plusp == true);
	      *sense_paths_gplus = Hitlist_push(*sense_paths_gplus,hitlistpool,(void *) path
						hitlistpool_trace(__FILE__,__LINE__));
	    }
	  } else {
	    debug3(printf("Approx case 2: tplus, geneminus\n"));
	    if ((path = Trpath_solve_from_ends(&(*found_score),&(*found_transcriptp),
					       trdiagonalb,pos5b,pos3b,trdiagonala,pos5a,pos3a,
					       /*tplusp*/true,querylength,/*query_compress_tr*/query_compress_fwd,
					       query_compress_fwd,query_compress_rev,
					       trnum,troffset,trhigh,chrnum,/*geneplusp*/false,/*want_lowest_coordinate_p*/false,
					       indelinfo,nmismatches_allowed,
					       max_insertionlen,max_deletionlen,
					       intlistpool,uintlistpool,univcoordlistpool,listpool,
					       trpathpool,pathpool,vectorpool,transcriptpool,method)) != NULL) {
	      assert(path->plusp == false);
	      *sense_paths_gminus = Hitlist_push(*sense_paths_gminus,hitlistpool,(void *) path
						 hitlistpool_trace(__FILE__,__LINE__));
	    }
	  }

	} else {
	  /* Skip.  Alignment does not go to the ends of the read */
	}
      }

      FREE(tplus_diagpairs);
    }

    /* Case for n_tminus_diagpairs == 0 already handled above */
    if (n_tminus_diagpairs > 0) {
      for (i = 0, k = 0; i < n_tminus_diagpairs; i++, k += 2) {
	debug3(printf("tminus diagpairs: %u and %u\n",tminus_diagpairs[k],tminus_diagpairs[k+1]));
	trdiagonala = tminus_diagpairs[k];
	trdiagonalb = tminus_diagpairs[k+1];

	pos5a = Genomebits_first_kmer_left(&nmismatches_a5,transcriptomebits,query_compress_rev,
					   /*univdiagonal*/(Univcoord_T) trdiagonala,querylength,
					   /*pos5*/0,/*pos3*/querylength,/*plusp*/false,/*genestrand*/0,
					   /*query_unk_mismatch_local_p*/false,/*kmer*/index1part_tr);
	pos3a = Genomebits_first_kmer_right(&nmismatches_a3,transcriptomebits,query_compress_rev,
					    /*univdiagonal*/(Univcoord_T) trdiagonala,querylength,
					    /*pos5*/0,/*pos3*/querylength,/*plusp*/false,/*genestrand*/0,
					    /*query_unk_mismatch_local_p*/false,/*kmer*/index1part_tr);
	  
	pos5b = Genomebits_first_kmer_left(&nmismatches_b5,transcriptomebits,query_compress_rev,
					   /*univdiagonal*/(Univcoord_T) trdiagonalb,querylength,
					   /*pos5*/0,/*pos3*/querylength,/*plusp*/false,/*genestrand*/0,
					   /*query_unk_mismatch_local_p*/false,/*kmer*/index1part_tr);
	pos3b = Genomebits_first_kmer_right(&nmismatches_b3,transcriptomebits,query_compress_rev,
					    /*univdiagonal*/(Univcoord_T) trdiagonalb,querylength,
					    /*pos5*/0,/*pos3*/querylength,/*plusp*/false,/*genestrand*/0,
					    /*query_unk_mismatch_local_p*/false,/*kmer*/index1part_tr);

	if (trdiagonala /*+qstart:0*/ < (Trcoord_T) querylength) {
	  /* Skip */

	} else if (pos5a == 0 || pos3b == querylength) {
	  /* Setting a first, b second */
	  trnum = EF64_trnum(&troffset,&trhigh,transcript_ef64,trdiagonala - querylength,trdiagonala);
	  chrnum = Transcriptome_chrnum(&transcript_genestrand,transcriptome,trnum);

	  if (transcript_genestrand > 0) {
	    debug3(printf("Approx case 3: tminus, geneplus\n"));
	    if ((path = Trpath_solve_from_ends(&(*found_score),&(*found_transcriptp),
					       trdiagonala,pos5a,pos3a,trdiagonalb,pos5b,pos3b,
					       /*tplusp*/false,querylength,/*query_compress_tr*/query_compress_rev,
					       query_compress_fwd,query_compress_rev,
					       trnum,troffset,trhigh,chrnum,/*geneplusp*/true,/*want_lowest_coordinate_p*/true,
					       indelinfo,nmismatches_allowed,
					       max_insertionlen,max_deletionlen,
					       intlistpool,uintlistpool,univcoordlistpool,listpool,
					       trpathpool,pathpool,vectorpool,transcriptpool,method)) != NULL) {
	      assert(path->plusp == false);
	      *antisense_paths_gminus = Hitlist_push(*antisense_paths_gminus,hitlistpool,(void *) path
						     hitlistpool_trace(__FILE__,__LINE__));
	    }
	  } else {
	    debug3(printf("Approx case 4: tminus, geneminus\n"));
	    if ((path = Trpath_solve_from_ends(&(*found_score),&(*found_transcriptp),
					       trdiagonala,pos5a,pos3a,trdiagonalb,pos5b,pos3b,
					       /*tplusp*/false,querylength,/*query_compress_tr*/query_compress_rev,
					       query_compress_fwd,query_compress_rev,
					       trnum,troffset,trhigh,chrnum,/*geneplusp*/false,/*want_lowest_coordinate_p*/false,
					       indelinfo,nmismatches_allowed,
					       max_insertionlen,max_deletionlen,
					       intlistpool,uintlistpool,univcoordlistpool,listpool,
					       trpathpool,pathpool,vectorpool,transcriptpool,method)) != NULL) {
	      assert(path->plusp == true);
	      *antisense_paths_gplus = Hitlist_push(*antisense_paths_gplus,hitlistpool,(void *) path
						    hitlistpool_trace(__FILE__,__LINE__));
	    }
	  }

	} else if (pos5b == 0 || pos3a == querylength) {
	  /* Setting b first, a second */
	  trnum = EF64_trnum(&troffset,&trhigh,transcript_ef64,trdiagonala - querylength,trdiagonala);
	  chrnum = Transcriptome_chrnum(&transcript_genestrand,transcriptome,trnum);

	  if (transcript_genestrand > 0) {
	    debug3(printf("Approx case 3: tminus, geneplus\n"));
	    if ((path = Trpath_solve_from_ends(&(*found_score),&(*found_transcriptp),
					       trdiagonalb,pos5b,pos3b,trdiagonala,pos5a,pos3a,
					       /*tplusp*/false,querylength,/*query_compress_tr*/query_compress_rev,
					       query_compress_fwd,query_compress_rev,
					       trnum,troffset,trhigh,chrnum,/*geneplusp*/true,/*want_lowest_coordinate_p*/true,
					       indelinfo,nmismatches_allowed,
					       max_insertionlen,max_deletionlen,
					       intlistpool,uintlistpool,univcoordlistpool,listpool,
					       trpathpool,pathpool,vectorpool,transcriptpool,method)) != NULL) {
	      assert(path->plusp == false);
	      *antisense_paths_gminus = Hitlist_push(*antisense_paths_gminus,hitlistpool,(void *) path
						     hitlistpool_trace(__FILE__,__LINE__));
	    }
	  } else {
	    debug3(printf("Approx case 4: tminus, geneminus\n"));
	    if ((path = Trpath_solve_from_ends(&(*found_score),&(*found_transcriptp),
					       trdiagonalb,pos5b,pos3b,trdiagonala,pos5a,pos3a,
					       /*tplusp*/false,querylength,/*query_compress_tr*/query_compress_rev,
					       query_compress_fwd,query_compress_rev,
					       trnum,troffset,trhigh,chrnum,/*geneplusp*/false,/*want_lowest_coordinate_p*/false,
					       indelinfo,nmismatches_allowed,
					       max_insertionlen,max_deletionlen,
					       intlistpool,uintlistpool,univcoordlistpool,listpool,
					       trpathpool,pathpool,vectorpool,transcriptpool,method)) != NULL) {
	      assert(path->plusp == true);
	      *antisense_paths_gplus = Hitlist_push(*antisense_paths_gplus,hitlistpool,(void *) path
						    hitlistpool_trace(__FILE__,__LINE__));
	    }
	  }

	} else {
	  /* Skip.  Read does not extend to the ends, so this method is not applicable */
	}
      }
      
      FREE(tminus_diagpairs);
    }
  }

  return;
}


#if 0
/* Replaced by Transcriptome_search_ends and Transcriptome_search_complete */
void
Transcriptome_search (int *found_score,

		      List_T *unextended_sense_paths_gplus, List_T *unextended_sense_paths_gminus,
		      List_T *unextended_antisense_paths_gplus, List_T *unextended_antisense_paths_gminus,

		      List_T *sense_paths_gplus, List_T *sense_paths_gminus,
		      List_T *antisense_paths_gplus, List_T *antisense_paths_gminus,

		      Compress_T query_compress_fwd, Compress_T query_compress_rev,
		      char *queryuc_ptr, int querylength,

		      Trcoord_T **tplus_stream_array, int *tplus_streamsize_array, int *tplus_diagterm_array,
		      Trcoord_T **tminus_stream_array, int *tminus_streamsize_array, int *tminus_diagterm_array,

		      int *mismatch_positions_alloc,

		      Indelinfo_T indelinfo, Mergeinfo_uint4_T mergeinfo,
		      int nmismatches_allowed, int max_insertionlen, int max_deletionlen,
		      Intlistpool_T intlistpool, Uintlistpool_T uintlistpool,
		      Univcoordlistpool_T univcoordlistpool, Listpool_T listpool,
		      Trpathpool_T trpathpool, Pathpool_T pathpool, Vectorpool_T vectorpool,
		      Transcriptpool_T transcriptpool, Hitlistpool_T hitlistpool) {
  
  /* Need a separate tr_found_score, because we don't want to rely on
     the found_score from prior genome search methods, which may have
     failed Transcriptome_remap */
  int tr_found_score = querylength;

  Trcoord_T *tplus_positions_5, *tminus_positions_5, *tplus_positions_3, *tminus_positions_3;
  int n_tplus_positions_5, n_tminus_positions_5, n_tplus_positions_3, n_tminus_positions_3;
  int tplus_diagterm_5, tminus_diagterm_5, tplus_diagterm_3, tminus_diagterm_3;

  debug(printf("Searching transcriptome ends\n"));
  search_transcriptome_ends(&tr_found_score,

			    &(*sense_paths_gplus),&(*sense_paths_gminus),
			    &(*antisense_paths_gplus),&(*antisense_paths_gminus),
				
			    &tplus_positions_5,&n_tplus_positions_5,&tplus_diagterm_5,
			    &tminus_positions_5,&n_tminus_positions_5,&tminus_diagterm_5,
			    &tplus_positions_3,&n_tplus_positions_3,&tplus_diagterm_3,
			    &tminus_positions_3,&n_tminus_positions_3,&tminus_diagterm_3,

			    queryuc_ptr,querylength,indelinfo,
			    query_compress_fwd,query_compress_rev,
			    nmismatches_allowed,max_insertionlen,max_deletionlen,
			    intlistpool,uintlistpool,univcoordlistpool,listpool,
			    trpathpool,pathpool,vectorpool,transcriptpool,hitlistpool);

  if (tr_found_score <= nmismatches_allowed) {
    /* Found paths */
  } else {
    /* Will reassign paths from search_transcriptome_complete */
    debug(printf("Searching transcriptome complete\n"));
    search_transcriptome_complete(&tr_found_score,

				  &(*unextended_sense_paths_gplus),&(*unextended_sense_paths_gminus),
				  &(*unextended_antisense_paths_gplus),&(*unextended_antisense_paths_gminus),

				  &(*sense_paths_gplus),&(*sense_paths_gminus),
				  &(*antisense_paths_gplus),&(*antisense_paths_gminus),
				
				  tplus_positions_5,n_tplus_positions_5,tplus_diagterm_5,
				  tminus_positions_5,n_tminus_positions_5,tminus_diagterm_5,
				  tplus_positions_3,n_tplus_positions_3,tplus_diagterm_3,
				  tminus_positions_3,n_tminus_positions_3,tminus_diagterm_3,
				 
				  queryuc_ptr,querylength,indelinfo,mergeinfo,
				  tplus_stream_array,tplus_streamsize_array,tplus_diagterm_array,
				  tminus_stream_array,tminus_streamsize_array,tminus_diagterm_array,

				  mismatch_positions_alloc,query_compress_fwd,query_compress_rev,
				  nmismatches_allowed,
				  intlistpool,uintlistpool,univcoordlistpool,listpool,
				  trpathpool,pathpool,vectorpool,transcriptpool,hitlistpool);
  }

  if (tr_found_score < *found_score) {
    *found_score = tr_found_score;
  }

  return;
}
#endif


void
Transcriptome_search_setup (int index1part, int index1interval_in, int index1part_tr_in,
			    Indexdb_T tr_indexdb_in, Transcriptome_T transcriptome_in,
			    EF64_T transcript_ef64_in, Genomebits_T genomebits_in,
			    Genomebits_T genomebits_alt_in, Genomebits_T transcriptomebits_in) {

  index1interval = index1interval_in;
  index1part_tr = index1part_tr_in;
  
  tr_indexdb = tr_indexdb_in;

  transcriptome = transcriptome_in;
  transcript_ef64 = transcript_ef64_in;

  genomebits = genomebits_in;
  genomebits_alt = genomebits_alt_in;
  transcriptomebits = transcriptomebits_in;

#ifdef HAVE_64_BIT
  leftreadshift = 64 - index1part - index1part;
  oligobase_mask = ~(~ (Oligospace_T) 0 << 2*index1part);
#else
  leftreadshift = 32 - index1part - index1part;
  oligobase_mask = ~(~ (Oligospace_T) 0 << 2*index1part);
#endif

  return;
}


