static char rcsid[] = "$Id: 77b355c6b8a5498715cf9700164400efff13f51a $";
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "transcript-remap.h"

#include <stdio.h>
#include "assert.h"
#include "uintlist.h"
#include "junction.h"
#include "transcript.h"
#include "repair.h"


/* Needed for speed when unspliced transcripts are possible */
#define INVALID_TRANSCRIPTS_SUFFICIENT 1

#ifdef DEBUG
#define debug(x) x
#else
#define debug(x)
#endif

/* Transcript_remap_matchp */
#ifdef DEBUG2
#define debug2(x) x
#else
#define debug2(x)
#endif


static Transcriptome_T transcriptome;
static IIT_T transcript_map_iit;


#define T Transcript_T


#if 0
static List_T
compute_exons_simple (int tstart, int tend, int *exonbounds, int nexons,
		      Listpool_T listpool, Transcriptpool_T transcriptpool) {
  List_T exons;
  char firstchar, lastchar;
  int exoni, exonj, i;

  debug(printf("bound_exon_simple with tstart %d, tend %d\n",tstart,tend));

  i = nexons - 1;
  while (i >= 0 && exonbounds[i] >= tstart) {
    i--;
  }
  exoni = i + 1;

  i++;				/* was i = 0 */
  while (i < nexons && exonbounds[i] <= tend) {
    i++;
  }
  exonj = i - 1;

  if (exoni == 0) {
    firstchar = '.';
  } else if (tstart == exonbounds[exoni - 1]) {
    firstchar = 's';
  } else {
    firstchar = '.';
  }

  if (exonj == nexons - 1) {
    lastchar = '.';
  } else if (tend == exonbounds[exonj]) {
    lastchar = 's';
  } else {
    lastchar = '.';
  }

  if (exonj == exoni) {
    exons = Listpool_push(NULL,listpool,
			  Exon_new(firstchar,exoni+1,lastchar,transcriptpool)
			  listpool_trace(__FILE__,__LINE__));
  } else {
    i = exoni;
    exons = Listpool_push(NULL,listpool,
			  Exon_new(firstchar,i+1,'s',transcriptpool)
			  listpool_trace(__FILE__,__LINE__));

    i++;
    while (i < exonj) {
      exons = Listpool_push(exons,listpool,
			    Exon_new('s',i+1,'s',transcriptpool)
			    listpool_trace(__FILE__,__LINE__));
      i++;
    }

    exons = Listpool_push(exons,listpool,
			  Exon_new('s',i+1,lastchar,transcriptpool)
			  listpool_trace(__FILE__,__LINE__));
    
  }

  return List_reverse(exons);
}
#endif



static Uintlist_T
compute_exons (Intlist_T endpoints, Univcoordlist_T univdiagonals, List_T junctions,
	       Univcoord_T chroffset, int querylength, Uintlistpool_T uintlistpool,
	       bool extend_qstart_p, bool extend_qend_p) {
  Uintlist_T coords = NULL;
  Chrpos_T chr_alignstart, chr_alignend, chr_left;
  Univcoord_T univdiagonal;
  Junction_T junction;
  int type;
  int endpoint, qstart, qend;

  debug(printf("Entering compute_exons\n"));

  /* Start first exon */
  univdiagonal = Univcoordlist_head(univdiagonals);
  chr_left = (univdiagonal - chroffset) - querylength;
  
  qstart = extend_qstart_p == true ? 0: Intlist_head(endpoints);
  chr_alignstart = chr_left + qstart + 1; /* 1-based */

  endpoints = Intlist_next(endpoints);
  univdiagonals = Univcoordlist_next(univdiagonals);

  while (univdiagonals != NULL) {
    junction = (Junction_T) List_head(junctions);
    if ((type = Junction_type(junction)) == INS_JUNCTION || type == DEL_JUNCTION) {
      /* Update chr_left for next iteration */
      univdiagonal = Univcoordlist_head(univdiagonals);
      chr_left = (univdiagonal - chroffset) - querylength;

    } else {
      /* Emit exon */
      endpoint = Intlist_head(endpoints);
      chr_alignend = chr_left + endpoint; /* Use previous value of chr_left; 1-based */
      coords = Uintlistpool_push(coords,uintlistpool,chr_alignstart
				 uintlistpool_trace(__FILE__,__LINE__));
      coords = Uintlistpool_push(coords,uintlistpool,chr_alignend
				  uintlistpool_trace(__FILE__,__LINE__));

      /* Start a new exon.  Revise chr_alignstart */
      univdiagonal = Univcoordlist_head(univdiagonals);
      chr_left = (univdiagonal - chroffset) - querylength;
      chr_alignstart = chr_left + endpoint + 1; /* 1-based */
    }

    endpoints = Intlist_next(endpoints);
    univdiagonals = Univcoordlist_next(univdiagonals);
    junctions = List_next(junctions);
  }

  /* Emit last exon */
  /* Intlist_head(endpoints) should be the same as Intlist_last_value(endpoints) */
  qend = extend_qend_p == true ? querylength : Intlist_head(endpoints);
  chr_alignend = chr_left + qend; /* 1-based */

  coords = Uintlistpool_push(coords,uintlistpool,chr_alignstart
			      uintlistpool_trace(__FILE__,__LINE__));
  coords = Uintlistpool_push(coords,uintlistpool,chr_alignend
			      uintlistpool_trace(__FILE__,__LINE__));

  return Uintlist_reverse(coords);
}
	

static Chrpos_T *
compute_exonends_geneplus (int *exonbounds, Chrpos_T *exonstarts, int nexons,
			   Vectorpool_T vectorpool) {
  Chrpos_T *exonends;
  int last_bound, exonlength;
  int exoni;

  exonends = Vectorpool_new_uintvector(vectorpool,nexons);

  last_bound = 0;
  for (exoni = 0; exoni < nexons; exoni++) {
    exonlength = exonbounds[exoni] - last_bound;
    exonends[exoni] = exonstarts[exoni] + exonlength - 1;
    last_bound = exonbounds[exoni];
  }

  return exonends;
}

static Chrpos_T *
compute_exonends_geneminus (int *exonbounds, Chrpos_T *exonstarts, int nexons,
			    Vectorpool_T vectorpool) {
  Chrpos_T *exonends;
  int last_bound, exonlength;
  int exoni;

  exonends = Vectorpool_new_uintvector(vectorpool,nexons);

  last_bound = 0;
  for (exoni = 0; exoni < nexons; exoni++) {
    exonlength = exonbounds[exoni] - last_bound;
    exonends[exoni] = exonstarts[exoni] - exonlength + 1;
    last_bound = exonbounds[exoni];
  }

  return exonends;
}


static void
bound_exon_geneplus (int *exoni, int *exonj, Chrpos_T chr_alignstart, Chrpos_T chr_alignend,
		     Chrpos_T *exonstarts, Chrpos_T *exonends, int nexons) {
  int i;

  debug(printf("bound_exon_geneplus with chr_alignstart %u, chr_alignend %u\n",
	       chr_alignstart,chr_alignend));

  i = nexons - 1;
  while (i >= 0 && exonends[i] >= chr_alignstart) {
    i--;
  }
  *exoni = i + 1;

  i = 0;
  while (i < nexons && exonstarts[i] <= chr_alignend) {
    i++;
  }
  *exonj = i - 1;

  return;
}


static void
bound_exon_geneminus (int *exoni, int *exonj, Chrpos_T chr_alignstart, Chrpos_T chr_alignend,
		      Chrpos_T *exonstarts, Chrpos_T *exonends, int nexons) {
  int i;

  debug(printf("bound_exon_geneminus with chr_alignstart %u, chr_alignend %u\n",
	       chr_alignstart,chr_alignend));

  i = nexons - 1;
  while (i >= 0 && exonends[i] <= chr_alignstart) {
    i--;
  }
  *exoni = i + 1;

  i = 0;
  while (i < nexons && exonstarts[i] >= chr_alignend) {
    i++;
  }
  *exonj = i - 1;

  return;
}


static List_T
remap_geneplus (int *overall_tstart, int *overall_tend,
		int *tstart_overhang, Chrpos_T *tstart_splice_distance,
		int *tend_overhang, Chrpos_T *tend_splice_distance,
		Intlist_T endpoints, bool extend_qstart_p, bool extend_qend_p, int querylength,
		Uintlist_T coords, int *exonbounds, Chrpos_T *exonstarts, Chrpos_T *exonends,
		int nexons, Transcriptpool_T transcriptpool, Listpool_T listpool) {
  List_T exons = NULL;
  Uintlist_T p = coords;
  char firstchar, lastchar;
  Chrpos_T chr_alignstart, chr_alignend;
  int exoni, exonj, i;
  int tstart;
  bool intronp = false, utrp = false;
  
  *overall_tstart = -1;

  *tstart_overhang = *tend_overhang = 0;
  *tstart_splice_distance = *tend_splice_distance = 0;

  while (p != NULL) {
    chr_alignstart = Uintlist_head(p);
    chr_alignend = Uintlist_head(Uintlist_next(p));

#ifdef DEBUG
    printf("exon 0: exonbound %d, exonstart %u, exonend %u\n",
	   exonbounds[0],exonstarts[0],exonends[0]);
    for (exoni = 1; exoni < nexons; exoni++) {
      printf("intron: %u\n",exonstarts[exoni] - exonends[exoni - 1]);
      printf("exon %d: exonbound %d, exonstart %u, exonend %u\n",
	     exoni,exonbounds[exoni],exonstarts[exoni],exonends[exoni]);
    }
#endif

    bound_exon_geneplus(&exoni,&exonj,chr_alignstart,chr_alignend,exonstarts,exonends,nexons);
    debug(printf("bounds geneplus: exoni %d, exonj %d\n",exoni,exonj));
    if (exoni <= exonj) {
      /* Overlaps an exon */
      debug(printf("First char: chr_alignstart %u versus exonstarts %u, exoni %d vs nexons %d\n",
		   chr_alignstart,exonstarts[exoni],exoni,nexons));
      if (chr_alignstart > exonstarts[exoni]) {
	/* Within exon */
	tstart = exonbounds[exoni] - (exonends[exoni] - chr_alignstart);
	firstchar = (p == coords) ? '.' : 'x';

      } else if (chr_alignstart == exonstarts[exoni]) {
	tstart = exonbounds[exoni] - (exonends[exoni] - chr_alignstart);
	firstchar = (exoni == 0) ? '.' : 's';

      } else if (exoni == 0) {
	tstart = exonbounds[exoni] - (exonends[exoni] - exonstarts[exoni]);
	firstchar = 'u';
	utrp = true;

      } else {
	/* Extends beyond exon */
	*tstart_overhang = exonstarts[exoni] - chr_alignstart;
	*tstart_splice_distance = exonstarts[exoni] - exonends[exoni-1] - 1;

	debug(printf("Computing tstart as exonbound %d\n",exonbounds[exoni]));
	tstart = exonbounds[exoni] + 1 /*1-based*/;
	firstchar = 'x';
      }

      if (*overall_tstart >= 0) {
	/* Skip */
      } else if (extend_qstart_p == true && *tstart_overhang == 0) {
	*overall_tstart = tstart + Intlist_head(endpoints);
      } else {
	*overall_tstart = tstart;
      }

      i = exoni;
      while (i < exonj) {
	debug(printf("%c%d%c\n",firstchar,i+1,'i'));
	exons = Listpool_push(exons,listpool,
			      Exon_new(firstchar,exoni,'i',transcriptpool)
			      listpool_trace(__FILE__,__LINE__));
	if (firstchar == 'i') {
	  intronp = true;
	}

	firstchar = 'i';
	i++;
      }

      p = Uintlist_next(Uintlist_next(p));

      debug(printf("Last char: chr_alignend %u versus exonends %u, exonj %d vs nexons %d\n",
		   chr_alignend,exonends[exonj],exonj,nexons));
      if (chr_alignend < exonends[exonj]) {
	/* Within exon */
	*overall_tend = exonbounds[exonj] - (exonends[exonj] - chr_alignend);
	lastchar = (p == NULL) ? '.' : 'x';

      } else if (chr_alignend == exonends[exonj]) {
	*overall_tend = exonbounds[exonj];
	lastchar = (exonj == nexons - 1) ? '.' : 's';

      } else if (exonj == nexons - 1) {
	*overall_tend = exonbounds[exonj];
	lastchar = 'u';
	utrp = true;

      } else {
	/* Extends beyond exon */
	*tend_overhang = chr_alignend - exonends[exonj];
	*tend_splice_distance = exonstarts[exonj+1] - exonends[exonj] - 1;

	debug(printf("Computing tend as exonbound %d\n",exonbounds[exonj]));
	*overall_tend = exonbounds[exonj];
	lastchar = 'x';
      }

      debug(printf("%c%d%c\n",firstchar,i+1,lastchar));
      exons = Listpool_push(exons,listpool,
			    Exon_new(firstchar,exoni,lastchar,transcriptpool)
			    listpool_trace(__FILE__,__LINE__));

    } else if (exonj < 0) {
      /* Skip: Before start of gene */
      debug(printf("%c%d%c\n",'u',0,'u'));
      exons = Listpool_push(exons,listpool,
			    Exon_new('u',0,'u',transcriptpool)
			    listpool_trace(__FILE__,__LINE__));
      utrp = true;
      p = Uintlist_next(Uintlist_next(p));

    } else if (exonj >= nexons - 1) {
      /* Skip: After end of gene */
      debug(printf("%c%d%c\n",'u',nexons,'u'));
      exons = Listpool_push(exons,listpool,
			    Exon_new('u',nexons,'u',transcriptpool)
			    listpool_trace(__FILE__,__LINE__));
      utrp = true;
      p = Uintlist_next(Uintlist_next(p));

    } else {
      /* Entirely in intron */
      tstart = *overall_tend = exonbounds[exonj];
      exons = Listpool_push(exons,listpool,
			    Exon_new('i',exonj,'i',transcriptpool)
			    listpool_trace(__FILE__,__LINE__));
      intronp = true;
      debug(printf("%c%d%c\n",'i',exonj+1,'i'));
      if (*overall_tstart < 0) {
	*overall_tstart = tstart;
      }
      p = Uintlist_next(Uintlist_next(p));
    }
  }

  debug(printf("\n"));
  
  if (extend_qend_p == true && *tend_overhang == 0) {
    *overall_tend -= querylength - Intlist_last_value(endpoints);
  }

  if (*overall_tstart < 0) {
    /* No overlap with any exon or intron */
    Exon_list_gc(&exons,listpool,transcriptpool);
    return (List_T) NULL;

  } else if (utrp == true) {
    /* Not consistent with transcript */
    Exon_list_gc(&exons,listpool,transcriptpool);
    return (List_T) NULL;

  } else if (intronp == true && List_length(exons) > 1) {
    /* Intron not allowed with other exons */
    Exon_list_gc(&exons,listpool,transcriptpool);
    return (List_T) NULL;

  } else if (Exon_list_consecutivep(exons) == false) {
    /* Not consistent with transcript */
    Exon_list_gc(&exons,listpool,transcriptpool);
    return (List_T) NULL;

  } else {
    return List_reverse(exons);
  }
}


static List_T
remap_geneminus (int *overall_tstart, int *overall_tend,
		 int *tstart_overhang, Chrpos_T *tstart_splice_distance,
		 int *tend_overhang, Chrpos_T *tend_splice_distance,
		 Intlist_T endpoints, bool extend_qstart_p, bool extend_qend_p, int querylength,
		 Uintlist_T coords, int *exonbounds, Chrpos_T *exonstarts, Chrpos_T *exonends,
		 int nexons, Transcriptpool_T transcriptpool, Listpool_T listpool) {
  List_T exons = NULL;
  Uintlist_T p = coords;
  char firstchar, lastchar;
  Chrpos_T chr_alignstart, chr_alignend;
  int exoni, exonj, i;
  int tstart;
  bool intronp = false, utrp = false;

  *overall_tstart = -1;
  
  *tstart_overhang = *tend_overhang = 0;
  *tstart_splice_distance = *tend_splice_distance = 0;

  while (p != NULL) {
    chr_alignstart = Uintlist_head(p);
    chr_alignend = Uintlist_head(Uintlist_next(p));

#ifdef DEBUG
    printf("exon 0: exonbound %d, exonstart %u, exonend %u\n",
	   exonbounds[0],exonstarts[0],exonends[0]);
    for (exoni = 1; exoni < nexons; exoni++) {
      printf("intron: %u\n",exonends[exoni - 1] - exonstarts[exoni]);
      printf("exon %d: exonbound %d, exonstart %u, exonend %u\n",
	     exoni,exonbounds[exoni],exonstarts[exoni],exonends[exoni]);
    }
#endif

    bound_exon_geneminus(&exoni,&exonj,chr_alignstart,chr_alignend,exonstarts,exonends,nexons);
    debug(printf("bounds geneminus: exoni %d, exonj %d\n",exoni,exonj));
    if (exonj >= exoni) {
      /* Overlaps an exon */
      debug(printf("First char: chr_alignstart %u versus exonstarts %u, exoni %d vs nexons %d\n",
		   chr_alignstart,exonstarts[exoni],exoni,nexons));
      if (chr_alignstart < exonstarts[exoni]) {
	/* Within exon */
	tstart = exonbounds[exoni] - (chr_alignstart - exonends[exoni]);
	firstchar = (p == coords) ? '.' : 'x';

      } else if (chr_alignstart == exonstarts[exoni]) {
	tstart = exonbounds[exoni] - (chr_alignstart - exonends[exoni]);
	firstchar = (exoni == 0) ? '.' : 's';

      } else if (exoni == 0) {
	tstart = exonbounds[exoni] - (exonstarts[exoni] - exonends[exoni]);
	firstchar = 'u';
	utrp = true;

      } else {
	/* Extends beyond exon */
	*tstart_overhang = chr_alignstart - exonstarts[exoni];
	*tstart_splice_distance = exonends[exoni-1] - exonstarts[exoni] - 1;

	debug(printf("Computing tstart as exonbound %d\n",exonbounds[exoni-1]));
	tstart = exonbounds[exoni-1] + 1 /*1-based*/;
	firstchar = 'x';
      }
      
      if (*overall_tstart >= 0) {
	/* Skip */
      } else if (extend_qend_p == true && *tstart_overhang == 0) {
	*overall_tstart = tstart + (querylength - Intlist_last_value(endpoints));
      } else {
	*overall_tstart = tstart;
      }
      
      i = exoni;
      while (i < exonj) {
	debug(printf("%c%d%c\n",firstchar,i+1,'i'));
	exons = Listpool_push(exons,listpool,
			      Exon_new(firstchar,exoni,'i',transcriptpool)
			      listpool_trace(__FILE__,__LINE__));
	if (firstchar == 'i') {
	  intronp = true;
	}

	firstchar = 'i';
	i++;
      }
      
      p = Uintlist_next(Uintlist_next(p));
      
      debug(printf("Last char: chr_alignend %u versus exonend %u, exonj %d vs nexons %d\n",
		   chr_alignend,exonends[exonj],exonj,nexons));
      if (chr_alignend > exonends[exonj]) {
	/* Within exon */
	*overall_tend = exonbounds[exonj] - (chr_alignend - exonends[exonj]);
	lastchar = (p == NULL) ? '.' : 'x';

      } else if (chr_alignend == exonends[exonj]) {
	*overall_tend = exonbounds[exonj];
	lastchar = (exonj == nexons - 1) ? '.' : 's';

      } else if (exonj == nexons - 1) {
	*overall_tend = exonbounds[exonj];
	lastchar = 'u';
	utrp = true;

      } else {
	/* Extends beyond exon */
	*tend_overhang = exonends[exonj] - chr_alignend;
	*tend_splice_distance = exonends[exonj] - exonstarts[exonj+1] - 1;

	debug(printf("Computing tend as exonbound %d\n",exonbounds[exonj]));
	*overall_tend = exonbounds[exonj];
	lastchar = 'x';
      }
      
      debug(printf("%c%d%c\n",firstchar,i+1,lastchar));
      exons = Listpool_push(exons,listpool,
			    Exon_new(firstchar,exoni,lastchar,transcriptpool)
			    listpool_trace(__FILE__,__LINE__));

    } else if (exonj < 0) {
      /* Skip: Before start of gene */
      debug(printf("%c%d%c\n",'u',0,'u'));
      exons = Listpool_push(exons,listpool,
			    Exon_new('u',0,'u',transcriptpool)
			    listpool_trace(__FILE__,__LINE__));
      utrp = true;
      p = Uintlist_next(Uintlist_next(p));

    } else if (exonj >= nexons - 1) {
      /* Skip: After end of gene */
      debug(printf("%c%d%c\n",'u',nexons,'u'));
      exons = Listpool_push(exons,listpool,
			    Exon_new('u',nexons,'u',transcriptpool)
			    listpool_trace(__FILE__,__LINE__));
      utrp = true;
      p = Uintlist_next(Uintlist_next(p));

    } else {
      /* Entirely in intron */
      tstart = *overall_tend = exonbounds[exonj];
      exons = Listpool_push(exons,listpool,
			    Exon_new('i',exonj,'i',transcriptpool)
			    listpool_trace(__FILE__,__LINE__));
      intronp = true;
      debug(printf("%c%d%c\n",'i',exonj+1,'i'));
      if (*overall_tstart < 0) {
	*overall_tstart = tstart;
      }
      p = Uintlist_next(Uintlist_next(p));
    }
  }

  debug(printf("\n"));

  if (extend_qstart_p == true && *tend_overhang == 0) {
    *overall_tend -= Intlist_head(endpoints);
  }

  if (*overall_tstart < 0) {
    /* No overlap with any exon or intron */
    Exon_list_gc(&exons,listpool,transcriptpool);
    return (List_T) NULL;

  } else if (utrp == true) {
    /* Not consistent with transcript */
    Exon_list_gc(&exons,listpool,transcriptpool);
    return (List_T) NULL;

  } else if (intronp == true && List_length(exons) > 1) {
    /* Intron not allowed with other exons */
    Exon_list_gc(&exons,listpool,transcriptpool);
    return (List_T) NULL;

  } else if (Exon_list_consecutivep(exons) == false) {
    /* Not consistent with transcript */
    Exon_list_gc(&exons,listpool,transcriptpool);
    return (List_T) NULL;

  } else {
    return List_reverse(exons);
  }
}


/* Returns repairs */
List_T
Transcript_remap (bool *found_transcriptp,
		  List_T *transcripts, List_T *invalid_transcripts,
		  Intlist_T endpoints, Univcoordlist_T univdiagonals, List_T junctions,
		  int querylength, Chrnum_T chrnum, Univcoord_T chroffset,
		  Uintlistpool_T uintlistpool, Listpool_T listpool,
		  Transcriptpool_T transcriptpool, Vectorpool_T vectorpool,
		  int desired_genestrand, int effective_sensedir,
		  bool extend_qstart_p, bool extend_qend_p, bool repairp) {

  List_T repairs = NULL;
  bool repairablep;
  Repair_T repair;
  int tstart_overhang, tend_overhang;
  Chrpos_T tstart_splice_distance, tend_splice_distance;

  Univcoord_T genomicstart, genomicend;
  Trnum_T trnum;
  int *matches, map_index;
  int nmatches, matchi;

  Uintlist_T coords;
  int nexons;
  
  int transcript_genestrand;
  Chrpos_T *exonstarts, *exonends;
  int *exonbounds;

  T transcript;
  List_T exons;
  int tstart, tend;
#ifdef DEBUG
  bool alloc_label_p;
#endif


  assert(transcript_map_iit != NULL);

  *transcripts = (List_T) NULL;
  *invalid_transcripts = (List_T) NULL;
  if (endpoints == NULL) {
    /* Can happen for fusion endpoints */
    return false;
  }

  debug(printf("Entered Transcript_remap with extend_qstart_p %d and extend_qend_p %d\n",
	       extend_qstart_p,extend_qend_p));
  debug(printf("endpoints %s\n",Intlist_to_string(endpoints)));
  debug(printf("univdiagonals %s\n",Univcoordlist_to_string(univdiagonals)));
  debug(printf("chroffset %u\n",chroffset));

  coords = compute_exons(endpoints,univdiagonals,junctions,
			 chroffset,querylength,uintlistpool,
			 extend_qstart_p,extend_qend_p);
  debug(printf("coords %s\n",Uintlist_to_string(coords)));

  genomicstart = Univcoordlist_head(univdiagonals) - querylength;
  genomicend = Univcoordlist_last_value(univdiagonals);

  /* Note: Specifying sortp == true would ensure that map_indices are
     in ascending order, but trnums may not be. */
#if 0
  /* Previously ignored desired_genestrand */
  matches = IIT_get_with_divno(&nmatches,transcript_map_iit,chrnum,
			       genomicstart - chroffset + 1 /* 1-based */,
			       genomicend - chroffset,/*sortp*/false);
#else
  if (effective_sensedir == SENSE_NULL) {
    desired_genestrand = 0;
  }
  matches = IIT_get_signed_with_divno(&nmatches,transcript_map_iit,chrnum,
				      genomicstart - chroffset + 1 /* 1-based */,
				      genomicend - chroffset,/*sortp*/false,
				      /*sign*/desired_genestrand);
  debug(printf("With desired genestrand %d, got %d matches\n",desired_genestrand,nmatches));
#endif

  for (matchi = 0; matchi < nmatches; matchi++) {
    map_index = matches[matchi];
    if ((trnum = Transcriptome_trnum(&nexons,&exonbounds,&exonstarts,transcriptome,map_index)) < 1) {
      /* Skip.  Not in transcriptome */
    } else {
      Transcriptome_chrnum(&transcript_genestrand,transcriptome,trnum);
      debug(printf("%d: %s with %d exons\n",
		   trnum,IIT_label(transcript_map_iit,map_index,&alloc_label_p),nexons));

      if (transcript_genestrand > 0) {
	exonends = compute_exonends_geneplus(exonbounds,exonstarts,nexons,vectorpool);

	assert(IIT_interval_low(transcript_map_iit,map_index) == exonstarts[0]);
	assert(IIT_interval_high(transcript_map_iit,map_index) == exonends[nexons-1]);

	if ((exons = remap_geneplus(&tstart,&tend,&tstart_overhang,&tstart_splice_distance,
				    &tend_overhang,&tend_splice_distance,
				    endpoints,extend_qstart_p,extend_qend_p,querylength,
				    coords,exonbounds,exonstarts,exonends,nexons,
				    transcriptpool,listpool)) == NULL) {
	  /* Skip */
	} else if (Exon_list_validp(&repairablep,exons) == true) {
	  debug(printf("Exon list is valid\n\n"));
	  transcript = Transcript_new(/*num*/trnum,transcript_genestrand,
				      tstart,tend,tstart_overhang,tend_overhang,
				      exons,nexons,transcriptpool);
	  *transcripts = Listpool_push(*transcripts,listpool,(void *) transcript
				       listpool_trace(__FILE__,__LINE__));
	  *found_transcriptp = true;

	} else if (repairp == false || repairablep == false) {
	  debug(printf("Exon list is not valid\n\n"));
	  transcript = Transcript_new(/*num*/trnum,transcript_genestrand,
				      tstart,tend,tstart_overhang,tend_overhang,
				      exons,nexons,transcriptpool);
	  *invalid_transcripts = 
	    Listpool_push(*invalid_transcripts,listpool,(void *) transcript
			  listpool_trace(__FILE__,__LINE__));

#ifdef INVALID_TRANSCRIPTS_SUFFICIENT
	  *found_transcriptp = true;
#endif

	} else {
	  debug(printf("(1) Exon list is repairable."));
	  debug(printf("  tstart overhang %d, tstart splice distance %u, tend overhang %d, tend splice_distance %u\n\n",
		       tstart_overhang,tstart_splice_distance,tend_overhang,tend_splice_distance));
	  /* Store transcript in case repair fails */
	  transcript = Transcript_new(/*num*/trnum,transcript_genestrand,
				      tstart,tend,tstart_overhang,tend_overhang,
				      exons,nexons,transcriptpool);
	  *invalid_transcripts = 
	    Listpool_push(*invalid_transcripts,listpool,(void *) transcript
			  listpool_trace(__FILE__,__LINE__));

#ifdef INVALID_TRANSCRIPTS_SUFFICIENT
	  *found_transcriptp = true;
#endif

	  transcript = Transcript_copy(transcript,transcriptpool,listpool);
	  repair = Repair_new(transcript_genestrand,tstart_overhang,tstart_splice_distance,
			      tend_overhang,tend_splice_distance,transcript,listpool);
	  repairs = Listpool_push(repairs,listpool,(void *) repair
				  listpool_trace(__FILE__,__LINE__));
	}

      } else {
	coords = Uintlist_reverse(coords);
	exonends = compute_exonends_geneminus(exonbounds,exonstarts,nexons,vectorpool);

	assert(IIT_interval_high(transcript_map_iit,map_index) == exonstarts[0]);
	assert(IIT_interval_low(transcript_map_iit,map_index) == exonends[nexons-1]);

	if ((exons = remap_geneminus(&tstart,&tend,&tstart_overhang,&tstart_splice_distance,
				     &tend_overhang,&tend_splice_distance,
				     endpoints,extend_qstart_p,extend_qend_p,querylength,
				     coords,exonbounds,exonstarts,exonends,nexons,
				     transcriptpool,listpool)) == NULL) {
	  /* Skip */
	} else if (Exon_list_validp(&repairablep,exons) == true) {
	  debug(printf("Exon list is valid\n\n"));
	  transcript = Transcript_new(/*num*/trnum,transcript_genestrand,
				      tstart,tend,tstart_overhang,tend_overhang,
				      exons,nexons,transcriptpool);
	  *transcripts = Listpool_push(*transcripts,listpool,(void *) transcript
				       listpool_trace(__FILE__,__LINE__));
	  *found_transcriptp = true;

	} else if (repairp == false || repairablep == false) {
	  debug(printf("Exon list is not valid\n\n"));
	  transcript = Transcript_new(/*num*/trnum,transcript_genestrand,
				      tstart,tend,tstart_overhang,tend_overhang,
				      exons,nexons,transcriptpool);
	  *invalid_transcripts = 
	    Listpool_push(*invalid_transcripts,listpool,(void *) transcript
			  listpool_trace(__FILE__,__LINE__));

#ifdef INVALID_TRANSCRIPTS_SUFFICIENT
	  *found_transcriptp = true;
#endif

	} else {
	  debug(printf("(2) Exon list is repairable."));
	  debug(printf("  tstart overhang %d, tstart splice distance %u, tend overhang %d, tend splice_distance %u\n\n",
		       tstart_overhang,tstart_splice_distance,tend_overhang,tend_splice_distance));
	  /* Store transcript in case repair fails */
	  transcript = Transcript_new(/*num*/trnum,transcript_genestrand,
				      tstart,tend,tstart_overhang,tend_overhang,
				      exons,nexons,transcriptpool);
	  *invalid_transcripts = 
	    Listpool_push(*invalid_transcripts,listpool,(void *) transcript
			  listpool_trace(__FILE__,__LINE__));

#ifdef INVALID_TRANSCRIPTS_SUFFICIENT
	  *found_transcriptp = true;
#endif

	  transcript = Transcript_copy(transcript,transcriptpool,listpool);
	  repair = Repair_new(transcript_genestrand,tstart_overhang,tstart_splice_distance,
			      tend_overhang,tend_splice_distance,transcript,listpool);
	  repairs = Listpool_push(repairs,listpool,(void *) repair
				  listpool_trace(__FILE__,__LINE__));
	}

	coords = Uintlist_reverse(coords);
      }
    }
  }

  FREE(matches);

  Uintlistpool_free_list(&coords,uintlistpool
			 uintlistpool_trace(__FILE__,__LINE__));


  *transcripts = Transcript_list_sort(*transcripts);
  *invalid_transcripts = Transcript_list_sort(*invalid_transcripts);
  repairs = Repair_make_unique(repairs,listpool);

#ifdef DEBUG
  printf("Returning transcripts ");
  Transcript_print_nums(*transcripts);
  printf("\n");
  printf("Returning invalid transcripts ");
  Transcript_print_nums(*invalid_transcripts);
  printf("\n");
#endif
  
#ifdef CHECK_ASSERTIONS
  Transcript_list_ascendingp(*transcripts);
  Transcript_list_ascendingp(*invalid_transcripts);
#endif

  return repairs;
}


static bool
matchp_exonends_geneplus (Chrpos_T donor_chrpos, Chrpos_T acceptor_chrpos,
			  int *exonbounds, Chrpos_T *exonstarts, int nexons) {
  Chrpos_T last_exonend;
  int last_bound, exonlength;
  int exoni;

  last_bound = 0;
  exonlength = exonbounds[0] - last_bound;
  last_exonend = exonstarts[0] + exonlength - 1;

  for (exoni = 1; exoni < nexons; exoni++) {
    debug2(printf("exon %d: comparing donor %u, acceptor %u with exonend %u, exonstart %u\n",
		  exoni,donor_chrpos,acceptor_chrpos,last_exonend,exonstarts[exoni]));
    if (last_exonend == donor_chrpos && exonstarts[exoni] == acceptor_chrpos) {
      return true;
    } else {
      exonlength = exonbounds[exoni] - last_bound;
      last_exonend = exonstarts[exoni] + exonlength - 1;
      last_bound = exonbounds[exoni];
    }
  }

  return false;
}

static bool
matchp_exonends_geneminus (Chrpos_T donor_chrpos, Chrpos_T acceptor_chrpos,
			   int *exonbounds, Chrpos_T *exonstarts, int nexons) {
  Chrpos_T last_exonend;
  int last_bound, exonlength;
  int exoni;

  last_bound = 0;
  exonlength = exonbounds[0] - last_bound;
  last_exonend = exonstarts[0] - exonlength + 1;

  for (exoni = 1; exoni < nexons; exoni++) {
    debug2(printf("exon %d: comparing donor %u, acceptor %u with exonend %u, exonstart %u\n",
		  exoni,donor_chrpos,acceptor_chrpos,last_exonend,exonstarts[exoni]));
    if (last_exonend == donor_chrpos && exonstarts[exoni] == acceptor_chrpos) {
      return true;
    } else {
      exonlength = exonbounds[exoni] - last_bound;
      last_exonend = exonstarts[exoni] - exonlength + 1;
      last_bound = exonbounds[exoni];
    }
  }

  return false;
}


/* Called by querystart_fusion_p and queryend_fusion_p*/
bool
Transcript_remap_matchp (Chrpos_T low_chrpos, Chrpos_T high_chrpos, Chrnum_T chrnum) {

  Trnum_T trnum;
  int *matches, map_index;
  int nmatches, matchi;

  int transcript_genestrand;
  Chrpos_T *exonstarts;
  int *exonbounds;

  int nexons;
#ifdef DEBUG
  bool alloc_label_p;
#endif

  if (transcript_map_iit != NULL) {
    debug2(printf("Looking for transcripts that match chrnum %d, low_chrpos %u, high_chrpos %u\n",
		  chrnum,low_chrpos,high_chrpos));
    matches = IIT_get_with_divno(&nmatches,transcript_map_iit,chrnum,
				 low_chrpos,high_chrpos,/*sortp*/false);

    for (matchi = 0; matchi < nmatches; matchi++) {
      map_index = matches[matchi];
      debug(printf("%s\n",IIT_label(transcript_map_iit,map_index,&alloc_label_p)));
    
      if ((trnum = Transcriptome_trnum(&nexons,&exonbounds,&exonstarts,transcriptome,map_index)) < 1) {
	/* Skip.  Not in transcriptome */
      } else {
	Transcriptome_chrnum(&transcript_genestrand,transcriptome,trnum);
	if (transcript_genestrand > 0) {
	  if (matchp_exonends_geneplus(/*donor_chrpos*/low_chrpos,
				       /*acceptor_chrpos*/high_chrpos + 1/*1-based*/,
				       exonbounds,exonstarts,nexons) == true) {
	    FREE(matches);
	    return true;
	  }

	} else {
	  if (matchp_exonends_geneminus(/*donor_chrpos*/high_chrpos,
					/*acceptor_chrpos*/low_chrpos + 1/*1-based*/,
					exonbounds,exonstarts,nexons) == true) {
	    FREE(matches);
	    return true;
	  }
	}
      }
    }

    FREE(matches);
  }

  return false;
}
   
    
void
Transcript_remap_setup (Transcriptome_T transcriptome_in,
			IIT_T transcript_map_iit_in) {
  transcriptome = transcriptome_in;
  transcript_map_iit = transcript_map_iit_in;
  return;
}

  
