static char rcsid[] = "$Id$";
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "repair.h"
#include <math.h>		/* For qsort */
#include "mem.h"
#include "path-eval.h"


/* Repair_path */
#ifdef DEBUG
#define debug(x) x
#else
#define debug(x)
#endif

/* Repair_unique */
#ifdef DEBUG1
#define debug1(x) x
#else
#define debug1(x)
#endif


#define T Repair_T


void
Repair_free (T *old) {
  FREE(*old);
  return;
}


T
Repair_new (int transcript_genestrand,
	    int tstart_overhang, Chrpos_T tstart_splice_distance,
	    int tend_overhang, Chrpos_T tend_splice_distance,
	    Transcript_T transcript, Listpool_T listpool) {
  T new = (T) MALLOC(sizeof(*new));

  new->transcript_genestrand = transcript_genestrand;
  new->tstart_overhang = tstart_overhang;
  new->tstart_splice_distance = tstart_splice_distance;
  new->tend_overhang = tend_overhang;
  new->tend_splice_distance = tend_splice_distance;
  new->transcripts = Listpool_push(NULL,listpool,(void *) transcript
				   listpool_trace(__FILE__,__LINE__));
 
  return new;
}
  

static int
repair_cmp (const void *a, const void *b) {
  T x = * (T *) a;
  T y = * (T *) b;

  if (x->transcript_genestrand > y->transcript_genestrand) {
    return -1;
  } else if (y->transcript_genestrand > x->transcript_genestrand) {
    return +1;
  } else if (x->tstart_overhang < y->tstart_overhang) {
    return -1;
  } else if (y->tstart_overhang < x->tstart_overhang) {
    return +1;
  } else if (x->tstart_splice_distance < y->tstart_splice_distance) {
    return -1;
  } else if (y->tstart_splice_distance < x->tstart_splice_distance) {
    return +1;
  } else if (x->tend_overhang < y->tend_overhang) {
    return -1;
  } else if (y->tend_overhang < x->tend_overhang) {
    return +1;
  } else if (x->tend_splice_distance < y->tend_splice_distance) {
    return -1;
  } else if (y->tend_splice_distance < x->tend_splice_distance) {
    return +1;
  } else {
    return 0;
  }
}


List_T
Repair_make_unique (List_T repairs, Listpool_T listpool) {
  List_T unique = NULL;
  T *array;
  int n, i, j;

  debug1(printf("Entered Repair_unique with %d repairs\n",
		List_length(repairs)));

  if ((n = List_length(repairs)) == 0) {
    return (List_T) NULL;

  } else {
    array = (T *) List_to_array(repairs,NULL);
    qsort(array,n,sizeof(T),repair_cmp);

    i = 0;
    while (i < n) {
      debug1(printf("Storing repair %d/%u, %d/%u\n",
		    array[i]->tstart_overhang,array[i]->tstart_splice_distance,
		    array[i]->tend_overhang,array[i]->tend_splice_distance));

      unique = Listpool_push(unique,listpool,(void *) array[i]
			     listpool_trace(__FILE__,__LINE__));
      j = i + 1;
      while (j < n && repair_cmp(&(array[j]),&(array[i])) == 0) {
	array[i]->transcripts = List_push_existing(array[i]->transcripts,array[j]->transcripts);
	Repair_free(&(array[j]));
	j++;
      }
      /* TODO: See if transcripts are in order */

      i = j;
    }
    FREE(array);

    Listpool_free_list(&repairs,listpool
		       listpool_trace(__FILE__,__LINE__));

    return List_reverse(unique);
  }
}


Path_T
Repair_path (int *found_score, T this, Path_T oldpath, int sensedir,
	     Compress_T query_compress_fwd, Compress_T query_compress_rev,
	     int nmismatches_allowed,
	     Intlistpool_T intlistpool, Univcoordlistpool_T univcoordlistpool,
	     Listpool_T listpool, Pathpool_T pathpool, Vectorpool_T vectorpool,
	     Transcriptpool_T transcriptpool) {
  Path_T path;
  List_T p;
  Transcript_T transcript;
  int querypos;

  path = Path_copy(oldpath,intlistpool,univcoordlistpool,listpool,
		   pathpool,vectorpool,transcriptpool);

#ifdef DEBUG
  printf("Entered Repair_path, transcript genestrand %d, with %d/%u and %d/%u\n",
	 this->transcript_genestrand,this->tstart_overhang,this->tstart_splice_distance,
	 this->tend_overhang,this->tend_splice_distance);
  Path_print(oldpath);
#endif

  if (this->transcript_genestrand > 0) {
    if (this->tstart_overhang > 0) {
      if (Intlist_head(path->endpoints) == 0 && Intlist_head(path->nmismatches) == 0) {
	/* Medial portion is still an exact match */
      } else {
	Intlist_head_set(path->nmismatches,-1);
      }
      /* Don't know if the overhang actually aligns to the previous exon */
      path->nmismatches = Intlistpool_push(path->nmismatches,intlistpool,-1
					   intlistpool_trace(__FILE__,__LINE__));
      
      if (Intlist_head(path->endpoints) == 0 && Intlist_head(path->ref_nmismatches) == 0) {
	/* Medial portion is still an exact match */
      } else {
	Intlist_head_set(path->ref_nmismatches,-1);
      }
      path->ref_nmismatches = Intlistpool_push(path->ref_nmismatches,intlistpool,-1
					       intlistpool_trace(__FILE__,__LINE__));
	
      querypos = Intlist_second_value(path->endpoints);
      /* No need to correct for Junction_ninserts on qstart side */

      if (this->tstart_overhang >= querypos) {
	Path_free(&path,intlistpool,univcoordlistpool,listpool,pathpool,transcriptpool);
	debug(printf("tstart_overhang %d >= querypos %d, so returning NULL\n",this->tstart_overhang,querypos));
	return (Path_T) NULL;
      } else {
	Intlist_head_set(path->endpoints,this->tstart_overhang);
	path->endpoints = Intlistpool_push(path->endpoints,intlistpool,0
					   intlistpool_trace(__FILE__,__LINE__));
      }

      path->univdiagonals =
	Univcoordlistpool_push(path->univdiagonals,univcoordlistpool,
			       Univcoordlist_head(path->univdiagonals) -
			       this->tstart_splice_distance
			       univcoordlistpool_trace(__FILE__,__LINE__));

      path->junctions = Listpool_push(path->junctions,listpool,
				      (void *) Junction_new_splice(this->tstart_splice_distance,
								   sensedir,/*donor_prob*/2.0,
								   /*acceptor_prob*/2.0,pathpool)
				      listpool_trace(__FILE__,__LINE__));

      Altsplice_free(&path->qstart_alts,pathpool);
      path->qstart_alts = (Altsplice_T) NULL;
    }

    if (this->tend_overhang > 0) {
      path->endpoints = Intlist_reverse(path->endpoints);
      path->univdiagonals = Univcoordlist_reverse(path->univdiagonals);
      path->nmismatches = Intlist_reverse(path->nmismatches);
      path->ref_nmismatches = Intlist_reverse(path->ref_nmismatches);
      path->junctions = List_reverse(path->junctions);


      if (Intlist_head(path->endpoints) == path->querylength && Intlist_head(path->nmismatches) == 0) {
	/* Medial portion is still an exact match */
      } else {
	Intlist_head_set(path->nmismatches,-1);
      }
      /* Don't know if the overhang actually aligns to the next exon */
      path->nmismatches = Intlistpool_push(path->nmismatches,intlistpool,-1
					   intlistpool_trace(__FILE__,__LINE__));
      
      if (Intlist_head(path->endpoints) == path->querylength && Intlist_head(path->ref_nmismatches) == 0) {
	/* Medial portion is still an exact match */
      } else {
	Intlist_head_set(path->ref_nmismatches,-1);
      }
      path->ref_nmismatches = Intlistpool_push(path->ref_nmismatches,intlistpool,-1
					       intlistpool_trace(__FILE__,__LINE__));
	

      querypos = Intlist_second_value(path->endpoints);
      if (path->junctions != NULL) {
	querypos += Junction_ninserts((Junction_T) List_head(path->junctions));
      }

      if (path->querylength - this->tend_overhang <= querypos) {
	Path_free(&path,intlistpool,univcoordlistpool,listpool,pathpool,transcriptpool);
	debug(printf("tend_overhang %d <= querypos %d, so returning NULL\n",this->tend_overhang,querypos));
	return (Path_T) NULL;
      } else {
	Intlist_head_set(path->endpoints,path->querylength - this->tend_overhang);
	path->endpoints = Intlistpool_push(path->endpoints,intlistpool,path->querylength
					   intlistpool_trace(__FILE__,__LINE__));
      }

      path->univdiagonals =
	Univcoordlistpool_push(path->univdiagonals,univcoordlistpool,
			       Univcoordlist_head(path->univdiagonals) +
			       this->tend_splice_distance
			       univcoordlistpool_trace(__FILE__,__LINE__));

      path->junctions = Listpool_push(path->junctions,listpool,
				      (void *) Junction_new_splice(this->tend_splice_distance,
								   sensedir,/*donor_prob*/2.0,
								   /*acceptor_prob*/2.0,pathpool)
				      listpool_trace(__FILE__,__LINE__));

      path->endpoints = Intlist_reverse(path->endpoints);
      path->univdiagonals = Univcoordlist_reverse(path->univdiagonals);
      path->nmismatches = Intlist_reverse(path->nmismatches);
      path->ref_nmismatches = Intlist_reverse(path->ref_nmismatches);
      path->junctions = List_reverse(path->junctions);

      Altsplice_free(&path->qend_alts,pathpool);
      path->qend_alts = (Altsplice_T) NULL;
    }

  } else {
    if (this->tstart_overhang > 0) {
      path->endpoints = Intlist_reverse(path->endpoints);
      path->univdiagonals = Univcoordlist_reverse(path->univdiagonals);
      path->nmismatches = Intlist_reverse(path->nmismatches);
      path->ref_nmismatches = Intlist_reverse(path->ref_nmismatches);
      path->junctions = List_reverse(path->junctions);


      if (Intlist_head(path->endpoints) == path->querylength && Intlist_head(path->nmismatches) == 0) {
	/* Medial portion is still an exact match */
      } else {
	Intlist_head_set(path->nmismatches,-1);
      }
      /* Don't know if the overhang actually aligns to the previous exon */
      path->nmismatches = Intlistpool_push(path->nmismatches,intlistpool,-1
					   intlistpool_trace(__FILE__,__LINE__));
      
      if (Intlist_head(path->endpoints) == path->querylength && Intlist_head(path->ref_nmismatches) == 0) {
	/* Medial portion is still an exact match */
      } else {
	Intlist_head_set(path->ref_nmismatches,-1);
      }
      path->ref_nmismatches = Intlistpool_push(path->ref_nmismatches,intlistpool,-1
					       intlistpool_trace(__FILE__,__LINE__));
	

      querypos = Intlist_second_value(path->endpoints);
      if (path->junctions != NULL) {
	querypos += Junction_ninserts((Junction_T) List_head(path->junctions));
      }

      if (path->querylength - this->tstart_overhang <= querypos) {
	Path_free(&path,intlistpool,univcoordlistpool,listpool,pathpool,transcriptpool);
	debug(printf("tstart_overhang %d <= querypos %d, so returning NULL\n",this->tstart_overhang,querypos));
	return (Path_T) NULL;
      } else {
	Intlist_head_set(path->endpoints,path->querylength - this->tstart_overhang);
	path->endpoints = Intlistpool_push(path->endpoints,intlistpool,path->querylength
					   intlistpool_trace(__FILE__,__LINE__));
      }

      path->univdiagonals =
	Univcoordlistpool_push(path->univdiagonals,univcoordlistpool,
			       Univcoordlist_head(path->univdiagonals) +
			       this->tstart_splice_distance
			       univcoordlistpool_trace(__FILE__,__LINE__));

      path->junctions = Listpool_push(path->junctions,listpool,
				      (void *) Junction_new_splice(this->tstart_splice_distance,
								   sensedir,/*donor_prob*/2.0,
								   /*acceptor_prob*/2.0,pathpool)
				      listpool_trace(__FILE__,__LINE__));

      path->endpoints = Intlist_reverse(path->endpoints);
      path->univdiagonals = Univcoordlist_reverse(path->univdiagonals);
      path->nmismatches = Intlist_reverse(path->nmismatches);
      path->ref_nmismatches = Intlist_reverse(path->ref_nmismatches);
      path->junctions = List_reverse(path->junctions);

      Altsplice_free(&path->qend_alts,pathpool);
      path->qend_alts = (Altsplice_T) NULL;
    }

    if (this->tend_overhang > 0) {
      if (Intlist_head(path->endpoints) == 0 && Intlist_head(path->nmismatches) == 0) {
	/* Medial portion is still an exact match */
      } else {
	Intlist_head_set(path->nmismatches,-1);
      }
      /* Don't know if the overhang actually aligns to the next exon */
      path->nmismatches = Intlistpool_push(path->nmismatches,intlistpool,-1
					   intlistpool_trace(__FILE__,__LINE__));
      
      if (Intlist_head(path->endpoints) && Intlist_head(path->ref_nmismatches) == 0) {
	/* Medial portion is still an exact match */
      } else {
	Intlist_head_set(path->ref_nmismatches,-1);
      }
      path->ref_nmismatches = Intlistpool_push(path->ref_nmismatches,intlistpool,-1
					       intlistpool_trace(__FILE__,__LINE__));
	
      querypos = Intlist_second_value(path->endpoints);
      /* No need to correct for Junction_ninserts on qstart side */

      if (this->tend_overhang >= querypos) {
	Path_free(&path,intlistpool,univcoordlistpool,listpool,pathpool,transcriptpool);
	debug(printf("tend_overhang %d >= querypos %d, so returning NULL\n",this->tend_overhang,querypos));
	return (Path_T) NULL;
      } else {
	Intlist_head_set(path->endpoints,this->tend_overhang);
	path->endpoints = Intlistpool_push(path->endpoints,intlistpool,0
					   intlistpool_trace(__FILE__,__LINE__));
      }

      path->univdiagonals =
	Univcoordlistpool_push(path->univdiagonals,univcoordlistpool,
			       Univcoordlist_head(path->univdiagonals) -
			       this->tend_splice_distance
			       univcoordlistpool_trace(__FILE__,__LINE__));

      path->junctions = Listpool_push(path->junctions,listpool,
				      (void *) Junction_new_splice(this->tend_splice_distance,
								   sensedir,/*donor_prob*/2.0,
								   /*acceptor_prob*/2.0,pathpool)
				      listpool_trace(__FILE__,__LINE__));

      Altsplice_free(&path->qstart_alts,pathpool);
      path->qstart_alts = (Altsplice_T) NULL;
    }
  }
    
  Path_eval_nmatches(&(*found_score),path,query_compress_fwd,query_compress_rev);
  if (path->score_within_trims > nmismatches_allowed) {
    Path_free(&path,intlistpool,univcoordlistpool,listpool,pathpool,transcriptpool);
    debug(printf("score_within_trims %d > nmismatches_allowed %d, so returning NULL\n",
		 path->score_within_trims,nmismatches_allowed));
    path = (Path_T) NULL;

  } else {
    /* Extend transcripts */
    for (p = this->transcripts; p != NULL; p = List_next(p)) {
      transcript = (Transcript_T) List_head(p);
      if (this->tstart_overhang > 0) {
	Transcript_repair_tstart(transcript,listpool,transcriptpool);
      }
      if (this->tend_overhang > 0) {
	Transcript_repair_tend(transcript,listpool,transcriptpool);
      }
    }

    path->transcripts = this->transcripts;

#if 0
    /* Doesn't handle the case where two different repairs are possible for a given path */
    path->invalid_transcripts = Transcript_remove_subset(path->invalid_transcripts,this->transcripts,
							 listpool,transcriptpool);
#else
    Transcript_list_gc(&path->invalid_transcripts,listpool,transcriptpool);
    path->invalid_transcripts = (List_T) NULL;
#endif

#ifdef DEBUG
    Path_print(path);
    printf("Exiting Repair_path\n");
#endif
  }

  return path;
}


