# Copyright 2021 The Debusine Developers
# See the AUTHORS file at the top-level directory of this distribution
#
# This file is part of Debusine. It is subject to the license terms
# in the LICENSE file found in the top-level directory of this
# distribution. No part of Debusine, including this file, may be copied,
# modified, propagated, or distributed except according to the terms
# contained in the LICENSE file.

"""Utility methods used in management module."""

from tabulate import tabulate


def print_tokens(output_file, tokens):
    """Print tokens (table formatted) to output_file."""
    headers = ['Token', 'Owner', 'Created', 'Enabled', 'Comment']
    table = []

    for token in tokens.order_by('owner'):
        table.append(
            [
                token.key,
                token.owner,
                token.created_at.isoformat(),
                token.enabled,
                token.comment,
            ]
        )

    output_file.write(tabulate(table, headers))
    output_file.write()
    output_file.write(f'Number of tokens: {len(table)}')


def print_workers(output_file, workers):
    """Print workers (table formatted) to output_file."""
    headers = ['Name', 'Registered', 'Connected', 'Token', 'Enabled']
    table = []

    for worker in workers.order_by('registered_at'):
        token = worker.token
        table.append(
            [
                worker.name,
                datetime_to_isoformat(worker.registered_at),
                datetime_to_isoformat(worker.connected_at),
                token.key,
                token.enabled,
            ]
        )

    output_file.write(tabulate(table, headers))
    output_file.write()
    output_file.write(f'Number of workers: {len(table)}')


def print_work_requests(output_file, work_requests):
    """Print work_requests formatted as a table to output_file."""
    headers = [
        "id",
        "worker",
        "created_at",
        "started_at",
        "completed_at",
        "status",
        "result",
    ]
    table = []

    for work_request in work_requests.order_by("created_at"):
        worker_name = work_request.worker.name if work_request.worker else "-"

        table.append(
            [
                work_request.id,
                worker_name,
                datetime_to_isoformat(work_request.created_at),
                datetime_to_isoformat(work_request.started_at),
                datetime_to_isoformat(work_request.completed_at),
                work_request.status,
                value_or_dash(work_request.result),
            ]
        )

    output_file.write(tabulate(table, headers))
    output_file.write()
    output_file.write(f'Number of work requests: {len(table)}')


def datetime_to_isoformat(date_time) -> str:
    """Return date_time.isoformat() or - if it was None."""
    if date_time is None:
        return "-"

    return date_time.isoformat()


def value_or_dash(value):
    """Return the value if it is not None or -."""
    if value is None:
        return "-"

    return value
