#!/bin/sh

set -e

SUDO="sudo"
SUDO_U="sudo -u"

if [ $(whoami) = "root" ]; then
    SUDO=""
    SUDO_U="runuser -u"
fi

APT_OPTIONS=""
if [ "$NON_INTERACTIVE" = "1" ]; then
    APT_OPTIONS="-y"
fi

verify_minimum_package_version() {
    package_name="$1"
    minimum_version=$(
        sed -Ene 's/.*# deb: //p' setup.cfg |
        awk "/$package_name >=/ { print \$3 }"
    )
    if [ -z "$minimum_version" ]; then
        return
    fi

    installed_version=$(dpkg-query -W -f'${Version}' "$package_name")

    if dpkg --compare-versions "$installed_version" lt "$minimum_version"; then
        echo "ERROR: you need $package_name >= $minimum_version" >&2
        exit 1
    else
        echo "INFO: you have $package_name $installed_version >= $minimum_version"
    fi
}

sanity_checks() {
    if [ -e debusine/project/settings/local.py ]; then
        echo "ERROR: You already have a configuration file (debusine/project/settings/local.py)" >&2
        exit 1
    fi

    if [ -e data/debusine.sqlite ]; then
        echo "ERROR: You already have a database file (data/debusine.sqlite)" >&2
        exit 1
    fi
}

install_packages() {
    echo ">>> Ensuring we have the required packages"
    packages="$(sed -Ene 's/.*# deb: (\S*).*/\1/p' setup.cfg)"

    if ! dpkg-query -W $packages >/dev/null; then
        echo ">>> Installing the required packages"
        $SUDO apt-get $APT_OPTIONS install $packages
    fi

    for package in $packages; do
        minver=$(sed -Ene "s/.*# deb: $package >= //p" setup.cfg)
        if [ -n "$minver" ]; then
            verify_minimum_package_version $package $minver
        fi
    done
}

setup_settings() {
    echo ">>> Installing a configuration file"
    cp debusine/project/settings/local.py.sample debusine/project/settings/local.py
}

setup_database() {
    echo ">>> Configuring postgresql user and databases"
    $SUDO_U postgres -- createuser -d $USER || true
    if ! $SUDO_U postgres -- psql -A -l | grep -q '^debusine|'; then
        $SUDO_U postgres -- createdb -O $USER -E UTF8 debusine
    fi
}

if [ ! -e debusine/project/settings/local.py.sample ]; then
    echo "ERROR: are you at the root of the debusine repository?" >&2
    echo "USAGE: ./bin/$0"
    exit 1
fi

case "$1" in
    "")
        sanity_checks
        install_packages
        setup_settings
        setup_database
        ;;
    install_packages)
        install_packages
        ;;
    setup_settings)
        setup_settings
        ;;
    setup_database)
        setup_database
        ;;
esac

