package org.bouncycastle.pqc.jcajce.provider.test;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.security.KeyFactory;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.SecureRandom;
import java.security.Security;
import java.security.Signature;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;

import junit.framework.TestCase;
import org.bouncycastle.pqc.jcajce.interfaces.SPHINCSPlusKey;
import org.bouncycastle.pqc.jcajce.provider.BouncyCastlePQCProvider;
import org.bouncycastle.pqc.jcajce.spec.SPHINCSPlusParameterSpec;
import org.bouncycastle.util.Arrays;
import org.bouncycastle.util.Strings;
import org.bouncycastle.util.encoders.Hex;
import org.bouncycastle.util.test.FixedSecureRandom;

/**
 * Test cases for the use of SPHINCS+ with the BCPQC provider.
 */
public class SphincsPlusTest
    extends TestCase
{
    // test vector courtesy the "Yawning Angel" GO implementation and the SUPERCOP reference implementation.
    byte[] msg = Strings.toByteArray("Cthulhu Fthagn --What a wonderful phrase!Cthulhu Fthagn --Say it and you're crazed!");

    public void setUp()
    {
        if (Security.getProvider(BouncyCastlePQCProvider.PROVIDER_NAME) == null)
        {
            Security.addProvider(new BouncyCastlePQCProvider());
        }
    }

//    public void testSphincsDefaultKeyGen()
//        throws Exception
//    {
//        KeyPairGenerator kpg = KeyPairGenerator.getInstance("SPHINCSPlus", "BCPQC");
//
//        kpg.initialize(new SPHINCSPlusKeyGenParameterSpec(), new RiggedRandom());
//
//        KeyPair kp = kpg.generateKeyPair();
//
//        SPHINCSPlusKey pub = (SPHINCSPlusKey)kp.getPublic();
//
//        assertTrue(Arrays.areEqual(expSha2Pub, pub.getKeyData()));
//
//        SPHINCSPlusKey priv = (SPHINCSPlusKey)kp.getPrivate();
//
//        assertTrue(Arrays.areEqual(expSha2Priv, priv.getKeyData()));
//
//        KeyFactory keyFact = KeyFactory.getInstance("SPHINCSPlus", "BCPQC");
//
//        SPHINCSPlusKey pub2 = (SPHINCSPlusKey)keyFact.generatePublic(new X509EncodedKeySpec(pub.getEncoded()));
//
//        assertTrue(Arrays.areEqual(expSha2Pub, pub2.getKeyData()));
//
//        SPHINCSPlusKey priv2 = (SPHINCSPlusKey)keyFact.generatePrivate(new PKCS8EncodedKeySpec(priv.getEncoded()));
//
//        assertTrue(Arrays.areEqual(expSha2Priv, priv2.getKeyData()));
//    }

    public void testPrivateKeyRecovery()
        throws Exception
    {
        KeyPairGenerator kpg = KeyPairGenerator.getInstance("SPHINCSPlus", "BCPQC");

        kpg.initialize(SPHINCSPlusParameterSpec.sha256_128f_simple, new RiggedRandom());

        KeyPair kp = kpg.generateKeyPair();

        KeyFactory kFact = KeyFactory.getInstance("SPHINCSPlus", "BCPQC");

        SPHINCSPlusKey privKey = (SPHINCSPlusKey)kFact.generatePrivate(new PKCS8EncodedKeySpec(kp.getPrivate().getEncoded()));

        ByteArrayOutputStream bOut = new ByteArrayOutputStream();
        ObjectOutputStream oOut = new ObjectOutputStream(bOut);

        oOut.writeObject(privKey);

        oOut.close();

        ObjectInputStream oIn = new ObjectInputStream(new ByteArrayInputStream(bOut.toByteArray()));

        SPHINCSPlusKey privKey2 = (SPHINCSPlusKey)oIn.readObject();

        assertEquals(privKey, privKey2);
    }

    public void testPublicKeyRecovery()
        throws Exception
    {
        KeyPairGenerator kpg = KeyPairGenerator.getInstance("SPHINCSPlus", "BCPQC");

        kpg.initialize(SPHINCSPlusParameterSpec.sha256_128f, new RiggedRandom());

        KeyPair kp = kpg.generateKeyPair();

        KeyFactory kFact = KeyFactory.getInstance("SPHINCSPlus", "BCPQC");

        SPHINCSPlusKey pubKey = (SPHINCSPlusKey)kFact.generatePublic(new X509EncodedKeySpec(kp.getPublic().getEncoded()));

        ByteArrayOutputStream bOut = new ByteArrayOutputStream();
        ObjectOutputStream oOut = new ObjectOutputStream(bOut);

        oOut.writeObject(pubKey);

        oOut.close();

        ObjectInputStream oIn = new ObjectInputStream(new ByteArrayInputStream(bOut.toByteArray()));

        SPHINCSPlusKey pubKey2 = (SPHINCSPlusKey)oIn.readObject();

        assertEquals(pubKey, pubKey2);
    }

//    public void testSphincsDefaultSha2KeyGen()
//        throws Exception
//    {
//        KeyPairGenerator kpg = KeyPairGenerator.getInstance("SPHINCSPlus", "BCPQC");
//
//        kpg.initialize(new SPHINCSPlusKeyGenParameterSpec(SPHINCSPlusKeyGenParameterSpec.SHA512_256), new RiggedRandom());
//
//        KeyPair kp = kpg.generateKeyPair();
//
//        SPHINCSPlusKey pub = (SPHINCSPlusKey)kp.getPublic();
//
//        assertTrue(Arrays.areEqual(expSha2Pub, pub.getKeyData()));
//
//        SPHINCSPlusKey priv = (SPHINCSPlusKey)kp.getPrivate();
//
//        assertTrue(Arrays.areEqual(expSha2Priv, priv.getKeyData()));
//
//        KeyFactory keyFact = KeyFactory.getInstance("SPHINCSPlus", "BCPQC");
//
//        SPHINCSPlusKey pub2 = (SPHINCSPlusKey)keyFact.generatePublic(new X509EncodedKeySpec(pub.getEncoded()));
//
//        assertTrue(Arrays.areEqual(expSha2Pub, pub2.getKeyData()));
//
//        SubjectPublicKeyInfo pkInfo = SubjectPublicKeyInfo.getInstance(pub2.getEncoded());
//
//        assertEquals(new AlgorithmIdentifier(NISTObjectIdentifiers.id_sha512_256), SPHINCSPlusKeyParams.getInstance(pkInfo.getAlgorithm().getParameters()).getTreeDigest());
//
//        SPHINCSPlusKey priv2 = (SPHINCSPlusKey)keyFact.generatePrivate(new PKCS8EncodedKeySpec(priv.getEncoded()));
//
//        assertTrue(Arrays.areEqual(expSha2Priv, priv2.getKeyData()));
//    }
//
//    public void testSphincsDefaultSha3KeyGen()
//        throws Exception
//    {
//        KeyPairGenerator kpg = KeyPairGenerator.getInstance("SPHINCSPlus", "BCPQC");
//
//        kpg.initialize(new SPHINCSPlusKeyGenParameterSpec(SPHINCSPlusKeyGenParameterSpec.SHA3_256), new RiggedRandom());
//
//        KeyPair kp = kpg.generateKeyPair();
//
//        SPHINCSPlusKey pub = (SPHINCSPlusKey)kp.getPublic();
//
//        assertTrue(Arrays.areEqual(expSha3Pub, pub.getKeyData()));
//
//        SPHINCSPlusKey priv = (SPHINCSPlusKey)kp.getPrivate();
//
//        assertTrue(Arrays.areEqual(expSha3Priv, priv.getKeyData()));
//
//        KeyFactory keyFact = KeyFactory.getInstance("SPHINCSPlus", "BCPQC");
//
//        SPHINCSPlusKey pub2 = (SPHINCSPlusKey)keyFact.generatePublic(new X509EncodedKeySpec(pub.getEncoded()));
//
//        assertTrue(Arrays.areEqual(expSha3Pub, pub2.getKeyData()));
//
//        SubjectPublicKeyInfo pkInfo = SubjectPublicKeyInfo.getInstance(pub2.getEncoded());
//
//        assertEquals(new AlgorithmIdentifier(NISTObjectIdentifiers.id_sha3_256), SPHINCSPlusKeyParams.getInstance(pkInfo.getAlgorithm().getParameters()).getTreeDigest());
//
//        SPHINCSPlusKey priv2 = (SPHINCSPlusKey)keyFact.generatePrivate(new PKCS8EncodedKeySpec(priv.getEncoded()));
//
//        assertTrue(Arrays.areEqual(expSha3Priv, priv2.getKeyData()));
//    }
//
//    public void testSphincsSha2Signature()
//        throws Exception
//    {
//        KeyPairGenerator kpg = KeyPairGenerator.getInstance("SPHINCSPlus", "BCPQC");
//
//        kpg.initialize(new SPHINCSPlusKeyGenParameterSpec(SPHINCSPlusKeyGenParameterSpec.SHA512_256), new RiggedRandom());
//
//        KeyPair kp = kpg.generateKeyPair();
//
//        Signature sig = Signature.getInstance("SHA512withSPHINCSPlus", "BCPQC");
//
//        sig.initSign(kp.getPrivate());
//
//        sig.update(msg, 0, msg.length);
//
//        byte[] s = sig.sign();
//
//        assertTrue(Arrays.areEqual(expSha2Sig, s));
//    }
//
//    public void testSphincsSha3Signature()
//        throws Exception
//    {
//        KeyPairGenerator kpg = KeyPairGenerator.getInstance("SPHINCSPlus", "BCPQC");
//
//        kpg.initialize(new SPHINCSPlusKeyGenParameterSpec(SPHINCSPlusKeyGenParameterSpec.SHA3_256), new RiggedRandom());
//
//        KeyPair kp = kpg.generateKeyPair();
//
//        Signature sig = Signature.getInstance("SHA3-512withSPHINCSPlus", "BCPQC");
//
//        sig.initSign(kp.getPrivate());
//
//        sig.update(msg, 0, msg.length);
//
//        byte[] s = sig.sign();
//
//        assertTrue(Arrays.areEqual(expSha3Sig, s));
//    }
//
//    public void testSphincsRandomSigSHA3()
//        throws Exception
//    {
//        KeyPairGenerator kpg = KeyPairGenerator.getInstance("SPHINCSPlus", "BCPQC");
//
//        kpg.initialize(new SPHINCSPlusKeyGenParameterSpec(SPHINCSPlusKeyGenParameterSpec.SHA3_256), new SecureRandom());
//
//        KeyPair kp = kpg.generateKeyPair();
//
//        Signature sig = Signature.getInstance("SHA3-512withSPHINCSPlus", "BCPQC");
//
//        // random should be ignored...
//        sig.initSign(kp.getPrivate(), new SecureRandom());
//
//        sig.update(msg, 0, msg.length);
//
//        byte[] s = sig.sign();
//
//        sig = Signature.getInstance("SHA3-512withSPHINCSPlus", "BCPQC");
//
//        sig.initVerify(kp.getPublic());
//
//        sig.update(msg, 0, msg.length);
//
//        assertTrue(sig.verify(s));
//
//        sig = Signature.getInstance("SHA512withSPHINCSPlus", "BCPQC");
//        try
//        {
//            sig.initVerify(kp.getPublic());
//            fail("no message");
//        }
//        catch (InvalidKeyException e)
//        {
//            assertEquals("SPHINCS-256 signature for tree digest: 2.16.840.1.101.3.4.2.8", e.getMessage());
//        }
//
//        try
//        {
//            sig.initSign(kp.getPrivate());
//            fail("no message");
//        }
//        catch (InvalidKeyException e)
//        {
//            assertEquals("SPHINCS-256 signature for tree digest: 2.16.840.1.101.3.4.2.8", e.getMessage());
//        }
//    }

    public void testSphincsRandomSigSHA2()
        throws Exception
    {
        SecureRandom random = new FixedSecureRandom(Hex.decode("7C9935A0B07694AA0C6D10E4DB6B1ADD2FD81A25CCB148032DCD739936737F2DB505D7CFAD1B497499323C8686325E4711E95F8A383854BA16A5DD3E25FF71D3"
            + "061550234D158C5EC95595FE04EF7A25767F2E24CC2BC479D09D86DC9ABCFDE7056A8C266F9EF97ED08541DBD2E1FFA1"));
        byte[] msg = Hex.decode("D81C4D8D734FCBFBEADE3D3F8A039FAA2A2C9957E835AD55B22E75BF57BB556AC8");

        KeyPairGenerator kpg = KeyPairGenerator.getInstance("SPHINCSPlus", "BCPQC");

        kpg.initialize(SPHINCSPlusParameterSpec.sha256_128f, random);

        KeyPair kp = kpg.generateKeyPair();

        Signature sig = Signature.getInstance("SPHINCSPlus", "BCPQC");

        sig.initSign(kp.getPrivate(), new FixedSecureRandom(Hex.decode("061550234D158C5EC95595FE04EF7A25767F2E24CC2BC479D09D86DC9ABCFDE7056A8C266F9EF97ED08541DBD2E1FFA1")));

        sig.update(msg, 0, msg.length);

        byte[] s = sig.sign();

        byte[] expected = Hex.decode("0e0ce237e144bd229938e1e8ca499889771a0f75d510cecfaf9f64d6890ecb49fa5068fc8e44188d30ed08a1df800234c191b8ffd1d3140db601bbe472dfd0aed8a4f81c25881e671f4b65fd18cad383762c6d55eba712b9c9857e12f4d816e439344056aeaad56a3e3e30fcb95385722dcc461a7ff40ec72e9ff8e062eba531a90ef265ebff1c36aac29a6cf079395de7f7f6d5f74387b781fb6e12a402cf471b06438c93b653fd91ef9edf6932c5694f623a3d0a42288a7524b297eb9b077d589b90d427f3767f793bfeb4574bac23187a2da6a20bdd5a2f2cad03b6f7ddd1411eae44559905c78d9bf01a7415e64c0602cfb4e91b74d8cd09236e777bdfc205082ecbb7046496d380cba5cdd7ae2c173ba5e3c70d74f572d40af642bf724258bec9a0996bfebc124fb793b06104e17a5861b9a710c2e7d68bee366609d345a89bc7a41ad980ccf0e9a4ef3522c7a1ac8009ef42845301e3c38718c3fa1a44d197aaeeae6e06974c2602948b277643481914985186d2c7679988d4e85eda3506f2658d5b32036fe222a4cdb5b04bc2c3a758a24693300bfa1652960f3482eda5165c75a3134e78091a8cbc146d24c7dc35eacd6d0dcc25646e70efb842f3e2b912775250d04c07934a120eba450768b067b7f0d3839aa8c4fec30085fcb244b37e965e42f0e38a103acec816c654293f558ebe39956379a673923c3c278e7493b6b32ece93db591eb36d74e0b24377761a8319b558434fc0682b809b09d5605db2b63f776e3549fca8abb3e357edc71cd5ceae8e17e3b3b004f08930079c7f5841cb68726f0452ee477c9eb82fef6d1721b7acfa23e2bc8eee22cf99e353b7540144db0cc9b08e34e9e90be77f170803346f9d850fb487bf967d62e83cffa1beff239214e5c413d539a4a03187ca3e69b8845984d1dc1b94c49a1eba940c8440b0f150acb5885334c125eb289bd14d3992df558dd15eab971db90c2bd290a3a215235daa889f6cc796f525f29be80244250de221b99cc78f89ccc41a24f73884d2c33527bae5c762cc8ad9cc71389564dfefc3cc3b1001bcb1168f5ec2bc025769f0f0a9cbbf1b621c79f19110af4fd3ba07833a20af483787c868cb133842d3cfbf50f105afbb15d01484362d49a59a4513f9e61191b49d52ae51e4c990690d360325c00789fcc5355f7cee7cc913b2414ab297e4527a1c10233a4cb01446d52e7ea6470e04fd0ae4448fd6f89e03c22b34a1f25341" +
            "89eba7f321d711a1d630d288fee27a8537f86ac04a725a3c836cf9e865b18ef3786cd70ac790011cde1ae43c810b8b09babbb560e38b283e38fd76462ae45452cbd832fc13c241e0a07e9ffa002ea955c40739413bd3eb2b788817cce124b9851751f610ee8ea0709446c2073b9055ee8a375ae60f89ad8f7c39df7da2563afa25d2d1618c9d23bbd15c6fa1c392242450a1356c6f70b8087cc5c41c5a67cb09b5eb48d04f67d3f182d943bb085c24a81b2a3119b8ece73ded8e503e6aaaf74b20fcdd0c53a7ca978c91e226a139d91c363385be663fd9c0c8aaa52ba3dd4a803af1dfc7e56ddb0b05a6e5c2cd41ab1711303877be2598d40c0b5aa79ac90046d05475d898ef19fc767b6132b3fab57ea0204dd9a34ac1b1e9629ec7ddf91e7ddcf705c5e58aa46bc685136d7471dbde1a8a7ef50e6c79d35e614b85248360bbb8b221a1946654165f342fb97c4ca6453f2e36486d5d156dbf9efa081986213fee79e4b57e71f0130e1f1eeba332fa6ff6c5e535d0ac8686126b06c11a3ced015f3369e5a9ccb8c1c02fbded7ce1ac1d13970d08f0da05964028f2b1ab11471d8aa8d1769b1c7a2c450ec17b6bb83e2c27cc98a5237692a86eeee790def9ec4673f9ea65923fc803bdb390bfd40cfd1e571b2b956313103c983c5088271309ced35309e305805c91de58023cdbafda8bf979ede9de8a5c76748363fcec1e1cb0a9972590dbaba122ae1223e7cc06808059decaf7128ae260f491c46241a07a1e1289469b57068b8075ba31dec8530aaf488fa38f585478459b60e2afddca9d26ec1ef3ceb9bafd173b586fc4ade33cdf8b7a7a2f753e78a635b36a50ed517ef11e0c1cfd2e18a992866297d91c321dd337c6215f0f91398cbef7eead63f0897b4f1859dc9869602818f671585066b5b7e59a5e2db155f8c709d0ed96dd223583c34a64c9d953066e227850562d82f25e01f3603f05234871f3e9e6c4a1b4354076e4c6063bd53161db8dc307718e69c9f1e58b1c5603404ce28d661262e6649a47999a280d5812873e187e3fdbe3c14ddfe14e5d6aa5728d6ce66cc6b6b74ecc9db00e4204383acb5ccbf072e3b7c4355406d38230a457afefb35c9b925f7165ad4a1ce76d46c2057a178a0162703b53baeeb1563596cf42555423b5954c766c31c1f208b1c56250bf8681de8d7d584f0d5f5382868f1b6c68f2441a7460b063b73313efbf2675d61b71dee2c56fce8962fbb7112abe484ce2befee0d2a8e39580d1588" +
            "5ab0ce33741362dcae3ddafc5f1b40c30374da06c643a0ecd14b87413114e862f0c2d84fc965f824e8736a84bd6e5beeddd024e09692da6a24eb36af7edbbf648159ed4ac47f7582357bfa1197f69c86acfaabc9de81c96323e412ee61355ee6cf7ced7136806ec02d640aa4806660b9ce0cdd3773da05c8c6203fbad229451f663fca720d940a440c45a7088c79bc37dbb7736a02593f24b4cfdbbae8dc37861627e1d08cf7cc0ca21392423a98ca30dd32907fb1d7ac6eb4eee21a1b07c894e3693937da1aab49496049ded508eb288747e46984c1f858307746927f52275561c61366fc97df661f62f224212fef8ea5f26d912714c62ac823942c64a8a80364ec855faaf18b810c16a7b6d4573e87e040331f6b540fc358deaa09035a3bd30acf8874edb491b8593e790a9f3549368f9996852b204a479ddbd53a570a51a47b16acc47fe31a1b13c84121e63861e4adae9c5e8808519b3ff96bafe6b320223e40875fce460be8110fc24f6899a691db6a04ea099b83bd3d144789f88be989f278154a6de697aae058fb94e7f05401b6c3f19fe0ce276965b4a5734916fd46a5a4b4af6e52e2d2854797b107926ba070403a3a63fa21b3262231e1fe7a4d78f6b7aecdd9fd0eedf077df55b32250be15e3133e3aa88901d7b1a0cd9d947beeb1b9ebee82e2d23169fbb7a220a88409844c8d9ee0c24af8504ab6c07cbb7b7f7456e4bce3b0ed90879d456724ee9a8f341f0da9541125d7166a6bce77fedbce826765c3b498813797217860e51128407a991ca713d79ec62a962a0cff17cea3f081a300d77352fc7b00b591d03b95da1d66c712ffff9c9f74c4087cfee42b1a895f737ca321cf87618d481fd2a0bd7e13295db9bf2821aef92ec5c20d0ff9f3c62f9cea497b0e6586b4dea91c967ae5606deb3f6699bc1eb6f9966218abec3b2f9ac1465cb771f60db38ab281d7ae12c7cd6f46badc96c2ee41c7e7df51af324b0cdce11db052de6b2ca0e1610a9e671e533b46a88c3ecae9d4c2a6992e53e6a5208b8a3f313a93002abcc9391476a43ed58d2ce4bc830d34599f8bf99ed9d389f13e7b6efada34b98f955cc0cda5998bb65e303bce4f032816b0464488bb8ff297aeedf5975faac1f07dd8113ca6cdf3e36574b95121656079ed29fa92c3fde4efcdba2adcbf1f75ea2494a1911796e8b15a0cab7a42335796572cff9208fdd413215cba7053ebe79a2a435179250f28f4ce8671b75219824ca56414f78d7d0dc3823" +
            "62b4a6b72f6dda2fcc735f460fa25073ab4f17596c4f623c2c00eafcfb934179ed30e598d36c190856e7c0bc359c44e2e4384e8e4d89cc5adf712a5be1083574154bfd3fae0764d1a63fd2ec1a44e2488e0a83afb61398fe3cce96d6b6a52bd8dddd022852872d92d184088fcdd7fade8bcd57e769823e97571457a544e42dac79a12863a973b3bd739820de3156e5b1690915a5fa052da7de4f4f16e91e1b03eb23a4b3b6b165da9ba312080f85cc0c40166170d306a64baafe4fd8b2723e60dd2d91b39495bab9ea897edf1400dfc68b3c926bcddb5069d04c6b0966d6f31acdacbdd26657d2cf9802cdff997236594e2494138ea1d1a25380c419930cd9fb63bd67700939fe265326504bbe283a51fdc5e493e75080451cef1b9a277fd0c7135a46b76b714569be13a103b93923811e707c300c87079f7559fec9eec168077908aa943e6de98dcb61b9b0fe802780651ab84ef7bb81e84e3e96733064fb56fa68154417d769c003a2b548ea0db06edde43b8bc1b43390a385c5d1fc94f7a1c2985ba7e1429643c850a5b6e8327cf82eb269882788ff9c0fcc64973871cc53906dc9487bbf343747e396d6176e7255a819c86584159a5da95166834336bcf63ceba9cf2f6e6462e1ec25194ef845386c9368ee39b3360091013ac72238f5fec47dd113f0f5262df9902b507e4bbab1dd82d73feef33ae7ffe38ac2a4115e90518867bb0f322b44c7bca29b606a814358ae0f8f00c0c323bfe2796fc5791d6e17c9645b758377bf27cb859063f9b59aebde3a5355b87b1edb98b28d072769c0a36f1b96e04464ce2800b6afa0d2efe10d2e6bf5d3d3c455d5ae71bd5e48398d74ff57f504c62225db65541e9eacd14ee777d63b40d949a056e96626d385a76890a6e8474801974b907f5be9b49fc0e5417428c9e55b54333c48a9225bc9d0dc2d62b9623df081ba8287781b202bc3ae54c442ce21accf499728b3cf218cf9f9f4bf8437b2098b70359076fb7ec5e7717b75e225c5098c8f4ce1e7baf53cdce6221b751c2e1d9290c2510b42529a08673e7205b1bb9db54d01ebcedbaeb04901e8bce909746205a81a478c5425ac2af9c91efc97f4d38fbd9bca085d7812fa099971f6675800ae045ea525c6779da2eae4bb5da0481cef61ec3ae551674d22500ee52d84e4962360d23bb04aaa49049a410539cc7364a9b0113e4028d3bbe3fde9cf88aa6887692b3a171a6da3afab5222bcf410ac33064c06170162ee086de93f99a67" +
            "9316091104d837c1805c864eef8733bafe1b8a5961eb1bb8071ed3354d84a23647bb2ddfa04e11b0d899a28a5b0bd0a192897ec71ff191ad3ebad97f36e9e82053fc5c6876446657051da14359bbf610dcaceade6f3b817512048cd5f4c06b6331f47e2b61362fa5861183cbdca80c974840ae150f8b3d3d4b51d36be297e6fa205bd62b306c2069c06491cfae027e30cfae790067468c452187ae596b354372442fcfebe5bb9a620037b4f8f0aef52f91688dfe5d506126e1e01712609757b08164464475e2a9891e216075b889f7cc6fdf389a423aedcdd2c350a655bfda1fd1709123510bfc13b3666a2d9f406b064664343cd035dc8a30b92575952d253a875b1b4df9c2d370a20cb41f4b5b0aa557824e43e753cf342be8fc03afd5c2ad3cbcf42d2389a28aaa805c18ddf917b81c2453b5aeb49e57420eb1f55d98391bab3574e80bc04626dd181f9bcd02456bae86b368abd90d43140411b052c23881af7f0b72775ce5dd57177209686dfbce7a6a01dbadb7dedd3176665400626c6d94fde97fdd4827fc13597a46e8c1c3e91fef4dafd749bcfd02b0bda800df7e3e142861e3297cb00d772d34ef512b70524524efa6607224b8863af18f87b571bd745b38582295ce0be8fe2208b5262f09f2ea302282c8aaedd768bae7ba842cd72e5422c06575a42f62b733eb511769442f1d263fd7ccd7d47deddf1c5fe0ab310886c1524e101ea25e5714dba3de9fcafe4fc19d065c170328d3039425e66afa61786c4dabf360000c331e1c563d3312910ab0fdc53586fe4d97315a5c01b74b4c01fbbf4b3c02a17d5c3029bccaba92338f16ea22c90a9d39c81107d49f6407de60f38db5d7722005187f3a8e47b98eb8852c41be717385c4a07c27f2b421e77b011e8080851bebd710b43fb978a2225a3f32e6d07bb4068fdbc24ef73d30e7928cd6937334a697c73ac8f1b53080209e887248b22ca199ba6572e20b062764c01485a315a0f7d21f9ed9ac03fff2fe3d118a9cdd219f8b61401bbb5d53a3d3e3a6688e5e0afbb72bb098c725972ad33ccde276cf7ecd314ed98ea933117de8c2bb4020b8fea5b8888de7db32c46ae3906675078fe031845979249b66d651f8865748d2c177e484bea48d01e0c6bf3570f4f16784c2cc6de743d2feed2fc96fa585d6dcbee5d4d2806d8c77cdc776e0a898c2b9d00520d6da5d7aa5fa0e2d80d3b9a27255c835e6012da88afa83bc6733f5e8b6bb84716765c331007107cf5d3d45700" +
            "6c65cbe2511966598305a93a1d8c23189517aebcc99a018f4a4b84175ec94e17f09253c92d5b9499853b069743382c1b115811b6f4c105cf5a89c7bf289ff5c49e5bc610ad8e80896a53b65984859f1d491e711ddb773bac7bb936b1418d76ace4180ce687e78586f8d40ef5a208623af4b6dfe404493ce9f4eb7a4aeddea9d5c8ee5d3507cf2fca96177d7ebcfcc72f34b75f82f926206ec6071ca028223eafc7b4d864bd75473a03d64806d7645cfd6e5a6cc88538be041e8b706ba2357f91f09260a449ebfe3626a105602683c915f59254d499312f15563d6e83e6434f744eda8863a4ce3f259ab5c4c2e54b4a3cd37810613ff7b435b8acc39045f9b30ac3b797961029d48360ac09a2a69f53e54b979b72800e84339e59c6ef044699c32c27465912ccf9f3b22670dd8c2fbcc4d33886106c62acdcc37cd09dd21d2fef3501f58b5faed58c5959fd991187e5af148984a98b5faa3def2906c072a80da57a20b6b638ed2ad2eaff2a35fb3d95671122b8a30fe417a07244206e31569f6e26129c3d38df10de077922c9163065d8abc82424f2d4f81f8791c7ae6ae98a0231f9743476013a57a6242ee55ed53175df4598eea498e9adba2f653b939e43fc9763b6dd7d9d542c72b3a5fb79e348352bb9d7e13a5a5b05e06d1a95fe73fe96ea8870b4cfb0376d5ad086827d724055425cc0bc5d8a938d141b4d7e51c4391be6b6b8ec0556efe25df4cd4cf347277df1b719164b0c78d2a9eae5a111068bc65c3695aa754b714f5e154e898bd0556c2c998d9a17c39c58b80fbd2e658dbba67923c112d46310023078820d21cdec30635e09464081dd8c1c1e773fffb6010e37ae0d1e3b040ca3f8e6b33fea17c2b250f8c3f67c28455301a3419a7b3f710b54cfbc394c3070656bdb327ab0602a48f73461eeacb1f0d7d42e27993c0daf22e15d023e62547a158cd02280a5417c67ce15b764abacb10bc0494fa38fcd5231f21b1617e545a0f2ab3a1601176dae55be8e9b225a21e7d9ae0ba9739f3d108dc7ba61fc0dd2b7c69d22982e6da7c70ecbfe4e65924471c3118103a752114bf6cb0643a50074e73660c6a4b6a31eb5e0672edf8a70378fdb4c00fc584dfce99cd41ea678b817dc4fd2a045201abfead8373b1f59811d358d473ee6d3ccb7903776b22b4ddd301b83e7a8c2f150a73fc8bb0aca0580606e46997e2072730d79fbf4d9e5a8cd3b656ee3c3a3ef3d20634238146115a779c8048226aa9f40c721cddc86068" +
            "9f9d06982cf8db9c49c6d47b3282cece3ec4302edcea1572ba252bbd85c916941e9351fd61f6b14888ada9ec76fbe43ca32c319d7408f683cfaaf2de45f87d73698798dc27706da96bb4f3eb7e77895fdacf13585341cdaec3f4486e18f3dd347165c6c30d439a3032b6bc7e0c29a5e11dfc07c399fcfed05623302eaee878bfb5bd13e084e581f8ce9bc6c5e2a618d526d42eb3df5d3b61f2a7ded674a15333d69765c610ba09347b611eb5ea80c43d98f002fcab7896a356a59d6cadcd5d6966eeecb3dca97a7fe40cfbca2b405654b81796d12e2427ec66f988c0d4eda6c8dc1db9f1c313ea2db4b2fa9b00ae8c1df0bfe0195e8be32fdd5c3b000a97c260a4067dfb56cfe00946a1bc75d85a5c4ef115d78ce21fe2bbfc58ea3bc7573d56552da48d0e54b194575cdca4b8b84ea5e76840436cb6c855a7dd485033f2784a5458c1cfb85a6ead9b1d8af18463df0c3c2e819a69df14ba754b24339db85243876a818e22243b7109f850d1a3869e4be1b0c8714bb7590876c6beca7e352368ba6443a5364a79b6129c7fb67739d6d7e0a62c466df033baa070ea33702238ad953c748cbe2ccb1b1cdd69dc335b4ef77732c0c449685040066583ce39c366438b7566d68eb425f9f7c1ae351bf16bee239a026f4d4e212f452093115b4b230a942df4a4d7dbbb6bb7936942775f0579a8a0294bbe6c8fd370d07d15a241cc1a09a99641b304fe53a3f68b9a5a0004aba6a8ad41bc8ab42777a8d2ad2f5287cd3d2f280295d27755661bb0149abe0e60ef977d2cbf48cbc61a4157d6b532a8c7eaf4140956dfa54d0ef23cc93ccf7828c89dfc74399bd59caa47b535542792c78511acbbc02a94dc536cb4536c3bcdf394c22486219f80472c604ba166d45e44c3fd07b0b42614923509c63bda749c2bcd5f07cce2607c0294e110294f351f0af5174fd18d558cbe7edc3ed2aa385074bfc60bcc446102951204550d87db5afdb6f1a5ea3adb667926df64cc5f34c0d407772f20290c43110f427b3a65a88955627b605de075632a7e3e35ab230f2ddba82d8aff08b362d238b96e9837d2b7805946e5d377ec6125d24c8ffddd0676c4dd7af67cb8f5d4f0ee6e2afff29a5e0797f5117f89aaf884edd920fe4b318e059759dce092f921b47933c30fb75046eba72ba6ab00fca9242ffb47aabfe4f566d585542b044b6e64404eb88d1c842a059b9b4389f3bb40bec8edfcac7bcef4f2a9a0727337dfcd7dba1ca7970579e9d1755789b" +
            "ea2e75244798f7408b2b86ec4a18ddd3352d9fb471361bb866866466d3055875014d0b5190298f68e7d27094bb129aa4237c6cba043a92df23e49d4ec3807b2a3829e5503859763e9384f8df5432873344e27138c6b3bd9258bcaa05363d0ba09cc29967a378b68d02178af1fd848449a265dbda8e798272545ead40e46718f155589bb9d4a698115113d558aaf5557608a32959ce587d1ca6ad18f8b58c1e10110acd34fb0c1acafb3d927164bde6d7275b97ff823fac1c711214b8f40dc1740c50c92613876024654c3a510974d58af133db3f75e9e25c46319152935f2b2f4f3403b3e42d57630e0ed00724b5f4210ff88416dd7f3de2ecdeedac68d5048daa3c8563a28da80fd85a9a22c39136b8908023c1f26b39d0ac0bc94d4b64380c01d021e342d3a99f1744d4a692bb1cc6916f8fc87159a619e454df13cddd4e5e8e7f63631d20eb5537815c591e82e0cededd0a3137bc092c057697b6c0dd7d960ee576fd2e422077e8050cfd499ffd8e2d77051d294cdc6660edfd466fb62eb110d5cb3eaa4f384194eb21fe82cb1ae421ba2728f7df2ca11d298d7195fc6d987deeec091399e939700e5650577a1e36fe781a7508fd9e285fd23798c369b766f40dcb13dff5e0fd396e6014f2a71f8b6ddc693f37df2ef09e09d1621957f78707cc49f192aa786c266b2265b14da16d8c52ad7f08d18ec7bf6a9c75512f56d63ec295768af8388e794555c85ca9c0a83c489cc703c4ae8682d096a16839dcc8f55d9adc77cb0659a815c56f74f2c4228de9e235a003cb3f15ebb21ac8722381d380e3363828e6c74c8d98d865ebe7c058b93bce5ec99074b04039061079a24535e5629bca18db94eed248cb579e3241aa69937d66e34f85c89f643ff6a165ebf568b9f21d79257a8c44d103e2f56518cae650afe89f1a776771da0197b3565cfb0dc2c428d25e37c85bffc555ab3f8c0e50aa1d5bea2d4ab5c0d0d711187cf783fce243123d132257b822cc717f8e585a68b189305a3e667324d01114c503200d8d8b4ac24e5bac9d25b6911cccf1732b7bee992ec7263304e01d9d4846221314d8046fb863ce2033ab8f880ac4c4b3c825b5adb50e7f9e218a24378338c0f159d72114e3da8b98c9d8f6869ab18c44deb0f513d5077b306e1f7a5fb0814df5ab80049e321348763d1be75ec91f3cf5a619ef5521333cc1505f896c3ff9a432531504437fec3cc2f42c4ce1c88631207c890821e3ae6fb857e20bf03b8d0af6de9932e" +
            "7ecbbf915e262df594d09362a436fd36e3d83079babdd5a4ecf95ea8d7a18b6d26942e5012b7ff6713e6bec82be257ced4d8c5962c659acca1b683f35683cfdc8e970ea141244fcf3519964dfabd6c1d11d13b2757803baabe8684baf253b645dbffb376930e595addb048836699f300379f01d776510c90f0aec725ca6e2fe481bef0213f5b68d431c4f8aaeb40653b77b4415a4e4c263c0855933228dc60123ee8da3a0b94c666a7aeb740032f4f3755ec6347af8aaedf85966357c3471f23b52bc325b447e8ac3ebea73b42b71631bb28d532d2346a713b5bbf09805bce56d263288d07a85000f86d436e9325f15656abba51fee50737096a86fd0715490c45d1d9395221eb5103798a7e34a074b26c0672ec688239d95d321fe307d5052c0f0994948ba2370a4daa2bc34870151bdfdef4d94e4cf370f657a271988afc43cac14a26e34bddd83d85c7c03a07c1c6a6c954480b756d37b18f4680780171682cd397967fc0abac6ff7bb3eb16803cd31e9509b2932408912c2d61c86da0ce7986c624c51dae53dac9d385b94af665292155b3c78b431aadb1b3ede70920b2e60bbc20ef28e82855aa76dcbe19c87981f21acb54313a49616c5d3fb7e83381d2ed2e01de610b7825f5d5eb2165f96c1804c69cafba9641c74c19cf04568cc101928cd95bb8c574033e17bfc237e3fb26806745e8888c06f848c8e87e89f361e6402a2716dade548c3efdfa8b829404891913940f729e1f53a1c0d232b548668baea2b3de7f8057f63c7ff813b6bb281af11a10871a2f09a0750eb4916585f70355ae73a74656410a5932bf2ff19788f588352b45965d0c5490fa6f381a4a82690fc24b7add73e4be921fdd12501defbc96a80981a0c08e674d3bdaa4c395bcb991a9420d8a04562704e0c57f1d32167fb2f5ec6298e4fc5be952d11eff013d9e4a4d00629a732c3d57970c92185758c37348af97369d597c76773ab2d2d104c6f431b9e8c61339c4841adda16fb1ee7d2d393225402f457ea8f46b989b0c497de6b230771d846c183f039e8fe9bfa86351ce1e69e900f48e1c47a94697688a92de33bc25f3a4ba649c1732b90f46695145fd2f18b72747e37a9f3a9d6aa92bc4909fd0dadf5c06e41d0ed028e94d2b9ba684cfad02e02c51ce77b160f57cc5b95148bc80146fec75b1d0149bc766c02005ba72fc69a82b6dee256d26b266bb263ada1d5dd30373b54137e79ffd0396bf89fe1e69e1a75e1cba05133c9a274c405d47eb" +
            "e464ef30b69aff4c7604b66d831b5533532b5d68175685b3a5fdae858bc8e89fb0977e86d3500a90d70f7b03bdacd18a3e093d73eb39c942a0085acb4e1ab7d1a7fc3def18c231ff6ce0ecac309778eb57ddc2078bd10e64196603d63228ff87ce5a166b0d1cccb77420f0b355166e22525e02b63bce6d285c4bab83c87b94bc7e74176c95d2a11acdfc35eb33634a0a629dbb9944049d98101b1f3a1c8e847c5fe47147ff30f61280c3053eaa65803b596861b0d19d6efbec3814d6b2c11d102ba5902d497c97c4f413ef66430b11c1d2a914e8203970ce48e35c20c6bb1fabf4b0b4a43e1bb9021f06ca27028a3fb7b40a5f8dc1ea304ad61131fb5129d36ff214a5b40c9646fcec7379289a58539d075983fbd19cae3453e6989d11eb83ab604e48b2083713a06bdd199bb6672b5f427632c96fc47610c72f3bd18205ecaa4b312708e6deb509534cfb24114b6777248f5695a4a3d3c52cad4fb4d606ac720f9d1157ee668a0ad0fab5be5f25078788af850bc385cbcaf15f4bf118ab731cb4dd6eac201355aa85dcc3fb0493412462f8962ab269536795595961105f322a74cd27c4cbd75931ec28d8695659ff59001fb4f864ffab2119da27ad55fb02a0cd4f37f0a193ab4fab5caa7fb492c2fe0f563461955ad56adf8bbcb43070b468bf2f04bb1c6543207be52d6afe3e864dac52af33d01095ba75d0f74007be0d954c4f0be3691dfafb6778ca055009e183ce7f763dd0ac7e57f89e76f39624cb5c92cbb4107a9425f7a8e32d328fbec847f6f5322098965d4a3452a564cdfb5596b3c03dbd908364a8b14882fcf01364d8334571dafdc6c5d36efe0bfd0530cc9cec25862b4d0550595f6b82143c589a603359e068a14dabebc89aeae1dd2b28b4b7c536581df1357d83971d52756fdbe592594a66f42fb87f109bdc94d71b314ce3ee9468d36ade0bde36fe074de27137018da682a03f4bbd5355d8aae252a8c31baec3fa2dbd29d56e87fe3f622f2225bb4ece7f016be7c90751f627665db8aef6cb79df1509df36d681d3a54ccffbbde2ad065bd41f98e19c55e692f69c5e59f9798dfe6f21cc7aac8525ed97000860ad1ed944e3d329a70071e11cc92c487d5549fc824f691f081896060365ba25c4ec3ba0e3e23c6ed6e5670ed6c03c8e186c5233ad9bd453a40d36a485a654d81929421b58a0440590bc0819c22d5cf0196f51a4dd4287559938311f5d8ef08613c20a008108dad28a50f9832ea5008536be2825f" +
            "df8717cac2b6e66b9288dfdbc170c55ccc2796b54278eef3ff678d395730d4b169a1c2f338f7bd2b971951762d0dbe0cbf4a9c6f17ee56f053e05f1ef09e8f7334c2ca02bec8243d6ce2c246cac8c92892128195861c9b545bcf19b5de7a3a913a527559f23a32db2d1ed592a1a51ca7a42405b36948be118ac1c4e52e16c7e96cd1755a26a58e2359d140535fafa67f99a78d91a4b68816302cb62736a3ab5277856f544eca38e999c862c00628d0651f6920cf328d9eb50851fed97a9a61e9da87eead55af4783b5fc034cec7f0446d561ea2af8f239e36a4715f9b9d5c787c080a474c95e5ab5faf950c667aa620ed0f5c02f65284441341cb5e53e4e4a70b69a530f2c24a8d5c7ebdd9bf4a5b763268930d11ca07839fd987b6bd35d15a9a0f8aa6a4c07c4c62838055d18805961eef5cda0fdf8bba4ecaa631719384855d4af279667d137fff7e0bbd5c109646b1527d6ad0c6d603b6c40638f93a4825290c088d09c6ef136e5376fb092662916d71c7f42f2ce623b524bad6773cc0f29755011a52b782aa09d3af017e7f734ab51c999a729faa1ed3bb26f7cccbc0a947a82af0fa81649cd871bf16251b85ac9c5567af87e5e32bf068982d70f754bf8a1dbe665cb42a0eb524a1e5ba4b7379ec30224f7e22bff3bd22c518142dedb639e123ab394729e1cd33c342e4d93b61109a821eaed36d61c963ef15bdd0c00e90f0153db269ff45f77bb74b91a47f73e4bea333ec5d120c0a3e28ad1638a2b30388cd3266a4f34f9493065d20066b886b9dbb58b59ac82cfbdc482a0a0a16dfc8a53e6761e77959d33f28a67696b663e0bcf8792e940e7de00211f804409a6800e52faf2f69166edbd907e088b7c928990749757118b98ecb30b1ca313d7cc96c01743d39851948cb655cdcc712a32ab6a7dd08b674fa40251c66cda10ad3ab7fe42c77cca065a0a34b851aa9cf9d84fbc9f10f39dada3cd118974d5e3e1d6a6e08d9b3ed604059f1010aa5f18d16f3060f3937a95caa356a4b4de421a331423eb3570419663bd3e5116933a6c60fe9364a3705afa9363da956779301cb49e0b11255607f2230119fba83629c92a19662fe56ebafe8071b021e4af6ac7e09173d74cc652d3e1ebf575de59b54b5678d047ed7298e8d2e9f891c7d930fb7bf9d105865dd8c56ce5c2ea6c62df5bc2c21e37379c0871e0c5de8068ede97cfcf668dbeaee6439fe20089d3e61edb828b62ef02d55fc3830645dcf834da2cabc03d5d6b4e59" +
            "3e777db804e32f347d204c460a1cacfdc78d0ce029c42124399f5d8b9ccef304a0aea9a3700a4137a133ffb4acd6c43b95fc527c919e34362623e91ef37dc9bc96be08f2fe97f2f078e735d1ced586743c1cc29f236cba0c52b4249e9aad18e2d2836eea9b5c80558297c75a97b83743b1637f42ec4f9eca608b8a7e8bc247e210d7e46f0fc425d680d48470449d8ee7cb6524ff59f3d0d73e369b383c49fc25b84800a552a360d6f8aa64ee6c89a0b7ac2ee19a65c9611ff273ce5abe3257d2673e7e3763cd86fd36aa6551a0df8c06693f0ebb55224d570adb4850149ab3b8d91b77a39fed84e434e738c1a2bc9cd8e53f22ff5b69d9e28fd6675ef34e157b292826069f4b5a910088c2089e1475232348a640e8b88445213346157e0ccc8d82f07114d510e160f9ea42bff7053e3760f676d7923bb094695bfb0602bc0fec28820f756909aadfb5e5bc1011f86235f7910292548c098fbbdd39a307d84222625fef691c9ac51e6a20f33dfa6596ebce4c9d7593f4c5fd516ce74a73a4c411d11ea93cccec245e4a8369ca23fca9b0b615d496239eea247d74645ac46987bf55c170a0d682d77a4c412c56d97534dfbfacbdba5e7992e8d3ccab827c9cd1d85b02c9894a72bf094b3a4e4c89886922ed08344577609a7b2c0beb9f4b2d65c7c6517b3ad10f77eb1bb8df3036bdcd4a9e550f359c4ebd0c5282db5a4d90316a073137440ff8ad5a36e15734ecb3ab1854567869ec96fdabde9e542972161e24dc9a6da002ba09b24b28144eb7e7f8e0aeb6b67f29434015a72d4ea28fdaed8533c7ec602c5900168d7c538222f1a77ba7cb7ce3852690045e9fe362c513bc2c78573ca8e303ccad326d829c5ae36a10c492f003db6aaf4df619756a105cccaafb1f43d4357a238929144ed2100a7a0d85441d6d8a5c20393169de055eaffffd9a487a6236cc685c01838ab88ea30f71e8c864c78f85032ea9244257b1d82911c422166750517bc8ebea165b6a9ee2a8dce61aa996637f186f5f6d2035cb4038404bb4c0aaa5de5d9a717a22d548d1b4abea504e1c302915f45ea11fa702955ef60d9fd35765a20c02b9b684e67fb18bd3cdc68edbb84c7996656acf9c818825a0dd975eb525916a9eba478b914a074f8525a67fd8d93c0ca13b880d579272f92752503ef1528c35e6990a2220dd6f8bed601d750c43d090cad116f02030afe77b3cdd27f2f123171c154734bfe961b15a4a4d14f191c1364f5913e100a80dd99e5e3f1" +
            "3590225c920a80e783012ada8e2e8412d260ec06db1e9fae67e75ea3641c220a0de13bcf6ac23c29d161fdcc883b612e438920d7cc3caeb6e4dc0cf723d3373814f63d937eed43571714c9c79e7983e9868cd9dc98daeb0b31f29e1984d1e5b5cea1a04bb43106a43a57cbf1395f0c4489e020f6b0e08e52607f5f550f132a8d785d00b3a2adccfadd34f89cc136528a5ad60d01bb81b43a51d4e8336f822b6fd5a38152c43ea61b560c098a11bbb5f0337f1a5a2abdfda323441f4bd5aeab3f5a08bad208a862f296532591089f4c5a5e376c6ba337e1bd3167fe891edd2656cb8fd5cad5418f46e4660e553c65ff183ca2c93f98acbcb91a5c4470bdfaf3adcffe47f5cd2e35bbd606bf2ad0b90972851cdac67694e7315c070ae5b914b89b40666ab2b5b7ea57f5f1ae4b282424d9bad1b9588d589f13b203e018fca24c10ebbff28d3553150500c8f18acf774ea997ed0e4ab7ce673378ee7c8f2a0e309f35969f33b804166da41a012d1e6149c7b00857ba7c4d4672a461a9965f1bdddefa3abef7afa4f23b6111ab435908fb156efc142fd612ddeb8b3c74398b48e67d57db8896d186ec1532628b3d6ed7d1e8ae6b24a4de67a51214733e288ab52bd0338a017c7bd7e117b6a145ebed32910ddaf06cdbebfda73c5b06179be0efd763176bf1d313a854a8640407b5f0fe4c9b9f49b2426bfc660a0545427d7e1a0967659105ea338a033af53db6cfe1be9d97c5b3004f36ad785c9eca8b6d0393059e083adb7ff0c0c244de3893273a63739898a55a9381d9f2d5c677b5c3a974706de3eda89bb7b3a1da61a9b334c8cd6ba3bfb6c07e4d36c2c387055fe42a03d24b19043e2d9cd6886257a324331fd788aff0fed2cb0c7cfca2a6cb9cb4e487b6bf173003ed42cf9bc5590df4737534f939ce176cf8bddb6cb158126145143c0e5a178fbf6101ee36756b8702ae5b3ea69ecefe6c59a5ea9436106f20bdd652a31619c908649e909ae6640f57e1e22c1d7b189a6606e1038d831c7adf0bdbb475abdcb5c20e5b01f5d6fa577d718d61c8c82b45ea265ff5f2c839a37001955967c22430ffafd3d70331a112ab860e684e322d076c0556f3e26db2d67445b645f115487ea28b779e4bdae5aedea20ca97a638ba9bc13dad691d448cd63581a02c00baf5a04fa7fab38a5f828a033556bc7747e260fa437c7427a1a963466f82e85f6dfe42ddcb4e6bc46d586948ca07100910a25d4bca64a4fa9a830454e04bd5927198e3ce" +
            "cdd6e8f6aed493d08580b76c4dfe3edeb7a0ef222562daaec593cbd9771b122bda8858dff3d61a361dc577fcc55a34d90c0710c0468364330bf9fc424e9da64df9a9f49771a53b3895b1e2fa4b1f38e393cdf0ae6fa905421ffbb24bb9bd8b387d66b4a17828191e4ed2579fcc75caf4058fc5f43e37128fe6ab39d1184ef73e39831acb1f69507db93d1428ee0912a3e31941e3b664fbd14cb69db9f760001031c7c06a0d91d43f01d94c3c842a2f22f87c7435dcfe615f383721acdeb51f2175cabd1a0b39e308106ea8307b3a5370fd393c3b5990b03cc586bb7995df692fd47cf359e4e5c988e96e4484f2c17e77f92d2f62694054926ec3e3c6b74a0346a5050a88cff7ee7a43049d04bf2658d189010f0487eed77b2a616a8d520895296e25c1dbedfc60e9bb724a0e2aaf9af3bfa13058851a0dae88c7d0f34b2d19bbf184cbacaf65d3f25a0429f89ccabff4eb948e35767d55fbb0e8db9e2be23897454d526ce9f2172b9eb787de5d0306168375ed9b925408fd4649713a0e9d1ce895b2513c461735914fc733ff1c24af0328fe06455b1317ce6e30a55b4e05a31d0ca1cd69c5b84efe89146f7aebabb587bcc017bdae9478dd7c3959e1cd8df749a35dfdb692544abb532f3e4dd7acab4cfbda043df107bb969b7f461afa3398f7bfa83a17e076c32b64ba5140248942caaae11ea47e2fdf8e357c264921eea9633e8f4865465e85c30c775b639f384b5374596da04935cc1218c5433f45838cb5631c3c4316eb522d0a76be3492f70f53c5b5102dcbe7444c6c6bbe4c3d3aea2d6cc28f4aba410e31ce4d50d4e9bf11e255dbab790524948172453fb0e478a6857eb472234a712d57878871cd25082f989e3293c05f36c6a57e5a1eb5f204105f1a2a8e0ce9d5433cd2f8d03d0578dccb5675d40bc26fa3b77e12b3338f53107351c6f1afd22f270d5dc79ea0c801a828b1dd0d9af7386860f4df438b967dd97d10edd4fcf9083bbe1b789fc32a0ddae2d907f67b394fbf6bf435d0c9fb42e3fb9b910170854cce6726b60ca03edc2d52b9d1114f0015fe16e8f6b5770c24c9b5f5d232d1a5d4296184b2b4d066c361d965bfc3d0b960a36ae988542597cf188561a2877d46d7800ede5c8c17fa3a9d125ee881db7c700d98e0131c6f3075fee95d438ecb245fe5953e6f59fb5f916e1068bb290405e3e13f422e9b114e89f2d490935c3804242f562906a0cc116b155cc7e9e1f6b89e2611f2203278be2e494902fae78" +
            "cfac3f13686ecd69d5cf94e525265a518836e570d99089e5b2a48bd4df092fbffccf3b9de71d60ef0a71b0647facc00b9e35268cee2e0b9dd5f6776e9e47cd6471edcbe086148bfa784d103c1eb5987facb22c1ca090340e1202d713680a536ec8fc35f1afa5c167963444a9f251b76cc7c06384c8713e9b4ddbc3844e345cfa7e87a723894a33bb22062f46dff42106907802ae43233a8e1d5e9020c3f4405bee715f6dc2f6e803488e18a0691f873ddb6e6205edf31e8812e34e3b6c755ef30a6cb55ede8e3ca5d33bd6d89a4c74909b134ca13e647c456f395a4bbbd0638c9590d9a205d1ca095ea6fe79cf49a67312187a43711794e2abe925e87d8c3f1fa51e4151dce0f1ae46ace428300a64aad33c3c718de53e886a0cd450a7469fd95e441e294bd98cae32bcc9911052d6db2ac4a93ca265b132b914207b7cbab7dc10cb0ba244fca82df5391366e36652657d8fd3d0e5e7dc20ff5dde19a49d4ba81b71b27e46395c3da79367c60614ab35092781ee4530112cded155e19a6a29e1304c116d228d939b766b90eaa9bdc1204fea158f1a6571d84f3cb7a43926b5598b9a2acf4149ecac5ae8e3e76ed873819c365407ecbe4b70ea9558b019cae8e85d68995942cbc75a0191ec40bdf4a1627003d21bfdc01ab15726f909adc190c602e38f6a4b0246e83d2373ab83d918f4be2761d9f5f5fd80a3b29dfa6dcef1eac6fd45dfb5ae027a307260948f4c3de06901f63ccc9a9684f0433ef0d926dc15f1654fed810ef202af1975fd796909158a7637d64644cbc40836f63de0f5c15045455b9cb3c06ceca78cfd5605c7a4312ec596ec39158aa2e6bfeb82178c533d23216adca522a0831a17d5f070e572a5a141b966929eb4f563f892e2d415d7268d7d48f0d217870dea13b2f7dc8af8191d7ae79f7c5a5c5a0bf0e098f6eed7cea9b4be589ec90a660b3c30fba4dcc1eee0ec38c7d141e35e823e5a36133d52c8f2d37c9f333393b50616871e4393d05c69b74f3125f142f58fe89c2c5aed1a7d02e258bc97f099eca3567b4137f278e925d45c99eabec64582b4e4e472cd4ce6de580243dfc6537b53cc0ba3e5be711908733e17ffd990ef5601e8e449b70ffe157dd35ba0fd4ae79c8b2e74e2f15a06c1ba9faa3276142a17aa1dd456bcfeebdcdbe9b0e6e6ed80d0838b2a30441204980b91a7bcc10dc489b30032c1dba1d230eceb38473d1e18f92640d6d35ef746389e5b4e7673f69223df3ce57a619928514f345" +
            "b0ec89b876062eebc1a71bf7f65a6c5fac1ebf9d6ec358e9770d9c62c63aa83dc88a92bdb9a185cc3179db278ebdd464d89e568f38062a693e2c1d98601ed2b8a79e951732f995b71e448f531ecc3182bd80d9c0047d228c01593e1f844c998d5c319a5e3840acbfca75680e2c118919eba3374144187c1d65ea6f9b2bc6c23863a16dde7702c4df8c4c458a5baa04c6bdcd80f0c6eb15ef86dcda7e3c9dad646fea094a664e8971e90b51eeea7d8f393df1e22b7946d8bc1a0f0f39c5a44364732d17ec59b1faae6668b3c76804a5c4635e134823fe82969a12e9a3104dd25b9b10109c1bc2dcf34dfc2dfb4346ad777c11fa1df6edac99697d5e1f7abcb1a951fe31c90c7d5ace6e7a2ce43890f87f2c131b858ffba2bab00b8d1bbf8bc6ccdd67b922c1dd2ec1b6ac7c55cc0eb0e3c18cf05f18188e80d3860750fd8e28ffd2cf1e6a2fbb2a3de81efbc6e1529eebcb531b882edcda378bae1dc769b58b96976684037d8342adfe92c72cc043073c47963be65377a98ee9dd346da0b5f87d7e42ff8d49011db53985c6f4741b8579ba173999ebdd1a7d4158eddd3a63168d06e986f073860eb253cba62be9049c4434f81e43498534ec56896fc017f77e5672ea627f41df4dd66fcbca0f7ff204352972523d3ac16f78e5c0f214e6f9a1f4509780997c77eb38a4b73e87db005cc7a0e7f69cbb36777a02a12c6a6f7ba901cfba9cd07e286b222c93868ad1fa7e8f38cfcb087810ec5e971487f4c34f2158d545c3fbe0d671a0210e2fb49282d9bc808a75eb4030f2a6f61d58b79ed68f3c2555919664c5c416c74ebf6ac8d17eb09430a364c425e86890c44043b4c4af23741df3c6f8cb3f459caeb88f9b3bc72b9e45c8989d5083a776128f2794f550122ce7dd6476a1e68baa09819aa3d50dbe286ccf07fe65687d551a5ed0fd659f60bdcb80c18fc4b69ce8a7d605d13459010f72a033d414a458c24688d2a35809bcea29bbac0c6729b058ca441aa689bb2642e3a21f08e6f3e5a1f3bbb9c31b3c090e45ff20d56a8647049ff5dc6646546296eae638fec3416a39bb15200b5f03cde0d029603a40f455d18813b58f1cae2f7db243ff638a6a87dedebffa1d646cf75b8113e029804c9de16a16f148e688d2a59a82c041a21cbdd8fb0023601c4553df73f90f9e71f37d2fe7f142b31cdd213effc4978976c880173bdc8e2eabf772e0be6a1a93130e89d826c1b7470f316e005fcece386950d3025347fe8a2050bab2b38e2" +
            "8af5fe8a0dbaf40ecee16515404b559a08824814f3721b907ce22afa48c0e6284a75ff6d924c95a716c5b620b3678f7d1fa678a601c892286dde7d40af0c2ac2bce842fd753317ad036e8130daa123fdde587c6f2a90cd258427ce091006c285fe75aea4a602893eb199f6698b43576542e0b22de1ae8d32865988054dc5f12aa759e709f0600085aba13914ccb989f03bcc7b9c4296b295f40982eba441f086148e768483a520e5447a43d021997a85c7bee3712cf53a67204b608ba0654b701a89efe6ca79d4d0195269c52d9e9144f3739a3fb2033c90f8608ea0eda4555abb5ef4892b73c19d526d8b92a5255f16c92052bb0b6cacb0e28e9618689fdd6648d57df34edaef5dc22bbaf6e04cea70e7d1bc5e36407b4ed363806440d2830dd28526409517c40b406153cea20021a9eeef6e5f0734ba85b438482b21bb8786eff343a4915dd5114b5651733d7155111d65cf500cb75f8612969c1af805e4e8fe811cc21930d4c62d120262a3ee57649dbaa4fee734304cbd92171357761ba1b4afce24162e8c9576229ab28e2ce6ed79d1a355b8e4b2c313d8e983712a0855b4a18ebd182f1b04712474442c1ac05edb60ffa1bb4504992feee974296f7c9ab17d1d02267310ac11c222bdd4498f97efb1fe4af21517f04ed7b125394d371f906e338acf664f40d578973d5eb0bfba466d9b5bf0539faa51a65412bb98ce8459137295da4f99882e463debc27302a5a3a8230e76f5d4449b48c74bd8c33f3c84b3bbdfc9c9ec49ee07843f6c3a37072073dd4a850926965ae7ee82570f2e6b6651e040c8b0290c8dd5b05d99a7db0c97b7dc1560ccb78d62204132b64bf2a0734568e5a2e12d01a114e795cdf57a87176d008a605be2207240a7e9ed32a1f1dda5285b7cad76bb58b929c3a4d939aa321fac50c35cce089835d9b487d87094dec60f50622989b69dde9944ce3749cff2c921bc3d465f9c71dce8eefa438df27499b24bfc5d33430cba8f63684e58cb180141751febd529609933eb74174251a3e4bbb684f13702925d18ce3e9833ecdd68e254d8be91601200c2e90108e2db3b8569669f2722779c074693cc94be953497132aec4ab5957195f72b872de7ef475dde8b081c8018c4a648667fa0f2834d2c78efd07ba12b3bcacc982cc4fb6b35c60ec4a283f011c0707d68dd8c26d11ac17011467b2f0d1ccb96183b94886eb6082ef40bdcf8d6f22182aa0a151408ccceb1f19cad31c3de3e4ca5caf47f8b5c98ebb" +
            "b7e819c6fb5d6eee1464a4476a7fed1bd99893b70491999fd1cf37fb4c5271a39e79d0732d9bbb99bdf3554f6976fc2f6fa80e87da36519e6ca75fe283e471499a96d42ae233fb74bb4af6566aebb058181890d123535cbdff236afad7a0cfa91ae0ca4458b8dbc464d664066a675036c0c8e22e5760150b934e9a9574e67067e98a87eba5a60ad29c48e669067bcdfe50a72b666540f1780b42da5d790c2456fb5d471c4f886c041b2f07298f3fb0f0d5c13eacbed1bfcd81de70ab9a8660bfac93d8ffcb649a1483c31eec4aa2e4dc0e1aecffd825b6220bc9e3da6f0de773a41666163485af2a20dfd916fb693c125a7baa69418eafd9e267eb0362f499a1d6a6b000bb176539e5577975589eb5bbb6a49bd3a736965bc19dc7a9e21a606940f812c7c762856f8c142d04c268bec64ada484c3e1d51c4422ad2c6da9d7b104aa83d06ff747bdbc20424a9dcae31f213134cf3fdad429a582ce5a487c2e210cac0fdee90174c765a30adc31d5c7447a5f1e1cf7433465b41864e0ea0ee4e5603f9fe5d42a513b22c5e92ac062a9c122fea5d8bab980819e2b7be491290cfd901d334501f8da6d9732b1f90d6b222f860a76190ff5cdcfd4bbf96e35ca877c54ad9b4a04af7e3d85e6d0cf12f0096bc7bd48c4a5a76f3b97dc1e6e699ee89d05a6cf2737fa9fa7cc0e14fc3fcd80216477f3246fe154eb1f340ec6ed1fa9128ff18119cb04b97b4d2797e8228d3ea00f8232bac6d4813dc8737c1585866d9b8847514fdd1816c1533f47010b0ffc223eb421b8d8894fd37265bbb1e8cffab2d5dab0f15b1dd3850e8c0cca04ea31cca39c9e0fbaa2735773be161db4f49ff6b8f15a792b13a855d24089b9e8ed5dab1b1dd9613a880fdf21b7c37167c09420c44e6098afecc0bed7784ad4f4463b8938f7e70c5abdbdb644de6fe4e549f8ad5a2d32f8c9f0decabbdef8cfdc2410de1d1e8a8b71b6577ffe419db28f07d1390c31f59169718595ad8a2ce733ccc9b88fb29f7207bcef8643827e7ae50807d7647ac1fd5708e460aeb4937ecb9b45ffac9235475e3b008ff966a74b6948f76bf116cbcc2fe42f9e1051b8b9cceab35dc2ff7024068dc87e17b17f7028f71c4b2bc582e53bce477b5babfc989169271cad42855a59ff19622e2583ccb6041dda88a32c3a88c554c2aa00108ce3dfd40f605e53ab49bc476f66af33be510e1231f3428b5dde49037b183e586396d1a8ad031bb8ae0bbe0fa1ff27813352c75b40ff857049" +
            "b5ee39ca0cb06781888295381d17270cb010f11088f17c87393aa3bcc8e6922f1c0b129687fa10fa9d6165faee81ded8b49132dbef5ea0cdc1ae78641094564bedc39eb67640c984c2bf8099dcfb0cf9a5738f17ba5faf1ddb9c6479d9723e44f109a405c1b65c7284bebc2a543af190b7cd6653dc564f7f93a191e3c9dd1469e376acbbb50457c9c72bad6001b3b291d07ae94b49a64a5c19f8f62d61b29d5aa8ceb12a67f1997097bea942dea71600c616a23d154737f3a6c98122ade6e3fb2b1972324e66bc7f2e3da822f2b090227d4158506ba44e051dc5cd14036100ca21e2e8d65549b5fc567e206cb6f8209f84111eb4fd24e2a3150d91a6d8bffbb6c002094aafe181c98a3db75b8c3b1c1c26ea278852d08d574662c4f8c5f78994140e22b74c9819c5335b354dd03bd2183a28a0457e1cf08018b2d5da481bdb27b3255cb6df1f4c9226b0787e7eb5675d665c51eeb6ee4eb895d60b66f12c856db5933800650ebffa62a51c6404e81c3e706ca1fc68c86e90bab45e4bfe6a9e81614542bce67d35782a05cf4d0defff44439d42f775445babf71fa5436f81abab374398378115b56625fab75e604f7da46f692ed68a007923fd75b8c436227e751c131d5a41301a2bbf24a4359e390c4bca01aada9ded491a27a80b27fc86e50b082492c5bc3b4a8c5bd6bdb7f76188510df80a047c315da724375f4af46fd494823e762d9e0364212886138f890503614d8683072bd925a73ee75889fd81f4a19a1083eb3f7b0df3c2b32cb51671f0cdf03cefb34a026cb7d637777a34e83a6f64d40e329fc697c342669178bf6c5f2a273c93fd95114d6c7f1b8e25d724db1a770badc9efd9068904d66830f8dea162f387d72b5ce1eccf31f81cfe8a621943f46aa86c4d2915b39f07f52704b18930c2b55d0685ea1bae891a5a19715f037a40f40deba8069f1414ea36530dd580647f819ced75254758daca684e5f11df7833c371955b741b1018b60a6fa57ea826a7d173238ab3da492fe4ee323409b4ffcaef9eb4113d5cc1916ecc3a09f57afdefbf55b8d6e71b7c08f38365b73dfc7ccc9e264842bb5807de24cb04ac6158607f828bea1bb756b728a9e871da630eb271e23ae3ffcd0519cf19cd78f9f048b6ab66fcba16468b46122c97f51e072a34d16f23bb7204b04034e7130deb92706d201f0f0160d8bed34a520b6095271f79b237d39c13f2f81b7f814c1fe8c164a9dc6742ec4995368cde68c1405996b16827ee4f6" +
            "aa93fe311ad4c79d4260");
        assertTrue(Hex.toHexString(s), Arrays.areEqual(expected, s));

        sig = Signature.getInstance("SPHINCSPlus", "BCPQC");

        sig.initVerify(kp.getPublic());

        sig.update(msg, 0, msg.length);

        assertTrue(sig.verify(s));
    }

    public void testSphincsDeterministicSigSHA2()
        throws Exception
    {
        SecureRandom random = new FixedSecureRandom(Hex.decode("7C9935A0B07694AA0C6D10E4DB6B1ADD2FD81A25CCB148032DCD739936737F2DB505D7CFAD1B497499323C8686325E4711E95F8A383854BA16A5DD3E25FF71D3"
            + "061550234D158C5EC95595FE04EF7A25767F2E24CC2BC479D09D86DC9ABCFDE7056A8C266F9EF97ED08541DBD2E1FFA1"));
        byte[] msg = Hex.decode("D81C4D8D734FCBFBEADE3D3F8A039FAA2A2C9957E835AD55B22E75BF57BB556AC8");

        KeyPairGenerator kpg = KeyPairGenerator.getInstance("SPHINCSPlus", "BCPQC");

        kpg.initialize(SPHINCSPlusParameterSpec.sha256_128f, random);

        KeyPair kp = kpg.generateKeyPair();

        Signature sig = Signature.getInstance("SPHINCSPlus", "BCPQC");

        sig.initSign(kp.getPrivate());

        sig.update(msg, 0, msg.length);

        byte[] s = sig.sign();

        byte[] expected = Hex.decode("e8a1883716841afecbd6f9bd8648bbf851dcf054572943cdc02dc9664ae096d1975ab97b76f18af4bad13de35202c7bf6cb93e96ecc21ef7ccdc80de902f9d21ed2520913104bc5d2159f6816e537dd5f0f8b418acd330ff0f2301a0336270f510f32d3a76f751d9e0c79b181d9a08f6e5f2263adc8a310491c2a3d3fcf1e9e4bcafeb91f73c4c4c45a08ead5fae61998fad2149748a409f9494fb3cfd326df717f2d89052c182eff2177725cc29cf3ed2c24ddc1ff20cb964b7e11aaee0651dab5c9d1863485cc9ae75ef253e097106d0c82007ec05e3d58b9237dc1feb4aeadaaf3a91a8a9cb4ee610814aafcd487472a08e09f363b46c8dcb56f9389144270fa9bb5b92088173724cb48c94f82d07989e6d0ef3cc42b80b651a48d06de9ecd01a075f2455ca068899281599bd2e85654e35775d714c21c5c04ae95de62be52acee64bc5a0febb7e9378174b6d71ec75fd38a06f7cb2ec4351c42e65d7cf8070ce93aa3ded7877c9d76b357d74d1085fb70b169ac9250ed734c1059dc8d1b58d5f55c24f191cd9ea7b235a8321140640c92ac593fcd0665ef80ed40813441cc0b1e502faf8cfdf89a000f1380a9f0d8b41bc53bea9d62e553ea97122bf152d21e2d1de982a386e4907d87449e7770c7430fbb7d356b2b61ca4405ab066a7c16064123830da8fcfc652cd4f36737a58e144909785f790273b7c4308282c957ec4bc9538441a1594dd656325455ed92a0a388044ebfb18f4994003ab1ee2c9507a9723f933176580d9f4b160aa0de77a9fe264dba93863d59c158abe8b2c13a50ebdb274d1fc6a38f4637cebec0e6e1c96e8ef9b8a0b6d11d1eeeee0b3fc1b1a8f931c029b77fdaa62f31441c0ba73dd10e88dcd514104c2ddc8bbfc1d511b72291be2c72471bb1b2855d39d39134061c00037ececfd4026bacb72481f845cda0b480ad1b487e4a0cb6fba9d14eb381cb7632c91b8c99fd2018c715357292c7901161782dd00fa8115c8a888cf117722ae4f89b4dccf35b3449cceb6c416bd8a5a76e45eb54e5c6fb0be101ff45e1e3f5ce89dc4f3045e927a72457d1e6c267c0ec9b8645eddd06b0950c674e4180a51f2f0d85277ef687b2cf5419cb77af41591feb4e430a6fe0c9863f8150d500d83559c15fec21d804eefcaf092ddb1e0b784200b58727cf47d217ebde152b8a9ca392dad03eb7556d881ad46981dd30d6129d1cc4993d857e953c5a7f8ecd9cc4eb1eecb424bf905" +
            "4948b7beb7b6d812e58a1070550dbd26a572e9a40e3de2a46ae62c8ae59904937b69ebe9b745097c62928048e93f6352f8be9654917b5719052122d23ec91f4b7880eaa3904db91202d11cc34692a6323b5cbf0dc9a62b9da9026f0ddd01c8b011696d67b5bdc08e8c3bc332a8d872cc540d5a9d9b5e79dfd11bbd012b580c6a7fe47a742f1e983bfd36e697a1657318039c8d323ded8758442fe6f39678cc03715adac630f244d42befd04ef33a4f7d9da9fd52bbb640a9d2394f2f963870917002399ae880ff6c1d45d05b3be1f961b6467f754e43373ab37611ed9641816d5b6b6a79e58a45afeed6d3c63ff0eb8f427820a36d12afd2e18c860367638c04edd7cfa7fdde76c28ed316f0201f9841c32646f9207bc4f032949316b3e98ccfdd4d4f90e9da10b6e2567c02194a32d36f0392ffc107027c0df8d7f778be8b421a0b44fa25a57fa9abfb277d87519b7b87cb31ffea177f87b9758d5dcb3e0e890db3b37da55d38985004ce04b7331b340b26774dce20a7029db2c3f1eed328ef99cc7de2a8d12d3ae540d156889729fdd86cb93d054997d202c369fc0397125a084b59af52a303b79e6e9f1b4a9d5b6640f7d70d5a1a96fc257b0b83d75ea2e29808ee98ebc4e75ba56959fbd2350c70ad9785c3dd01fa586c7bce84d0b5af37b38641877a7975472666a9da857642884e2cb89caa6e54c54f3979b93a42e9364e279686049914f5e3a38ee9665d2189dce6bee50df27f16e1b428703f2b7c4ef14e158b2d0ffee1d041edcd2be997946842a6033cd17f1353e2df201b65ab87ed0c61e1448eb16f4e034aa28e8a74ee7c3b0994a2005371691491483004391f796b08950fc1c4476cbf96ef95d76848ee8b43a0fb0ccf95e4477d5589ba7fcd8aa6e508010e609ae5c011f7e3da70ae9ac659adbd22dec4dfe0ae79306f1ae4a4274e0a40f9956ead784b1031b9a26e8a94b3affbae374ee863859765269cfda6b4e6e0123cebd0810bc02120f2cabaf8a34bae014da3f44d7278063ed7673437df405c59fbc9e956e4a1eb6ae46e61b972b41855bb5918dc1fe2266a1d650d6bb86b4c27e4eb3a4aec51a0387acb327fc2799b0c974b0e877bd10207ae5e0603d5c454b298fb50b45a8fe11f7d55a545d606e891547d9a42b4ab4241157ab259be231fdb75ee134bb091f8e8b22ce7f4cd2e8f2ea869d56fa3e6526f6be160f4f314b916d98bf5eb34f8f0227f527a8e1145ee507e8c8c0f181a3efa6615254e0f10a" +
            "41da885a7e801a733c24a0fa24dea81705f62dcb3d05ead981739476c33b5e77d44ea85b2182749d29ae4cd98a18654979a5b087c3485bdbc0656b821d5a75555cf8ad190e140f38cf83d1337c207c89d5e315e2ba6eed8cb005168de6422512a4bbc6e862a97760c4e8c712f7971d2b416dd5fa016194d28b785367b3607f3d1c3523035e964cb28aea7f32c09a1da78a876b643ad0fd24d4a7255a23aaed45ecc540beb7bf2cc371a2e13c4e33fe84fba9b607365126c6b75270757e513a8a5b5910b24c2b43bd7d93be42563430be20641462b9306491bc0ebc467f3553cdd1e1b9b1c8940bf60eff7a58d2ef4688edc77ea677481c028de4511209c5a845beb66d75a7b5bf7505f99fde0e3d64cff1fb69adb85077e4c177b7908de5e1fae70ce3f335575fb5d52eb07c96d9f3cf84eb4fa8d709026e837d0b17aef401c5dbfd5157985fb52b9358e8bf2f0ee2d05ea15b633df2154e7ebaacb6184324faa0136d521d9d06f4b8ce2dea422cbe5ab6ba3e7efa2f2c921ffe6818f71d7e7ac79706d7eec35821b5cc3d6819e5b08ab356a1cde0e7eb61de9e64931ef54c181b46cf1938f6c5176819b3262f34d0d767faf39f8642dee2ebace43763a09283b79cef58b2c94acbab1b97d224b71a6e398cd193becd2b9485e6a504c75ccfc7aa7ee472b111b78eade4b169b518cc4dc69d7347284f09539131fabe0e82a667c53f661670c558775cab483800294d78904d8d65abc1b8838f5795658934805790f56dddb0aff336b12698806e375fb15549820a2e1aa627297ac382e789d8e05c35bc0a2b973f890ef39d5752f9ccac4010bf29d4f56b5d9fb188b4ecfcbc0603ff8d23b74ff4fd5c96a37581e1fc1b4d5d599ec602d5890d5b35757e0403817c430ccbb3e700e660d76bab51cd2f96ccc0a8c1b28515ea966c67acca36265dcbc44080a56ec610d15cc274f6b8f3bc3d9a7d4b1c2475f7baaabff3288817c77f10da84c9c400611beb9077047aa05bdc1701870cda7c5e3bac4b789cd9a8e4a05ca27f7d5a4a669faa14e6a4e21bb7541ef030f735a33c04889dd5252a54ebad18dfabf398ef3fea693aa46af12f5671cf7e032aa558288797c4d8173f8d18a8364cbaf7a17b7031a97909368dbeabf9b7cb6338b7879ce000d912f9745580ecf7478fecd38babc9aa1529d7e191b4506559188a3525d93861724318a4a17f9377f799358f502e54fc402d32407e1d5b2f40611660af5affe47b92343b3207b54347a" +
            "4a4a3ee4a71b0004d0dde7f4ac6bdae5759b994c2388436eb47afaaef694a7c77f2b4399177940cc4b534a5f524c97555fe101a2818842fdde0df24c01c1f07bba6a403e3a53ccf7f4bec3369e034c97b7cc0f3f520691899579486af481ebc7858181ece98093b6bbf9fbdfc7c721f9bfe7d181c85493f861bc9b2f8795dbad1348d2e8a0c6fd67674ce232f098491b12d1f1d6726a5d275ff52f5f259336309360cda2b0f450bfa9686f76de5e7ad9ac23b7e9785713d0ed46fd41f7c96c99cb684c4137bf9a3a4570c6999014b42538b1ab8704b5c2cf9aabab2e37ad80f963d6e00821e55feb7ef22ce3f7037efc38b735cbe8c52d7f4847652c13074e6864504f5370414d19500169b1b0934a7a51860bdb9b91904b70a8c3dfef84fdadb6dca6ef92da0c79b10fdcd4372ea33fdc0f434927cb66b1b8663cf8fa455824b71573d068efdba6239021d66fb666ee66e95dc0efd7be727e6ed1976a903bdac2ffb7fc7fc98798db4bdab2d6c3aab572f8b77792e0fd9993d1743b25da43b1b1ae06db30330c54e07b7edbd387753d8f32ba253281b345cedace071f05c82850562b6d256e493fbbcdd598437f3c193a46ad201c673cc8d8aaf2b3d9ec6601120699e747b1fc85e62efe188f0d943f8d8a30e90214ea4b9c0cb0e89f94ca0bfd7157d75f76eef56550287d9ac900f5890c2f95c062375c499c1d55e58078396fe9a65f72ddb27d51df4ef23ed8ee4889da78149e8a46f16b2e5f57d602a5015280b3b100a472d7bbf68995dd21da7300eef386d05c7319910b817723ca6ee23e5a7da60ef5067006e66f021bf0d16c68fd074b555da462db2422414c5683ae2b2923fd880593d0ae32b1fb35326417013373bfa4c5f83104e07069a99176741b7d7be79b5cf3f07daece2a3bc62557e35dfca0a0701212689aacfb5987ea9aed3a796db6568b3955797b3b022d6b46e0fac18f40c89f1d3d1f92f7abe77e8d370499d5eb36624b9cee144da18bb0755d39f40fa9beef1e7f82db68b364606bb8a30d4445e28d3de76ca03a23379e75f276de8b7ba3e39c3dd46cdef427070f542bce2785f6b555c1a65e362211f67ded39052a504f16ba3e4d8999b6b68efc4e6bf0e812ff222bccfe34babd96532e102faccf4744fe81d14cd900698a877a1bc0fdcdaea38d2878551e1a081ba333fde860c006d9269f5a9a013ad485cd288633a5b298acd430bc030961e2025f7d5c83fed7d46d90bfce159c7effc872141688dfa" +
            "50370f245afdc45afb159cd03cb4ae6ed09d264970cb498d32e08c112da173fe54c978e8493d6d28f219854fdf8e451744a62f06e6d247d583361da24ea13c2ef6f54caff95267a82a4c7419f46897d3285cbb0783f4a5cebfb1c8df42e516d3664f5d660f4c5c7f798f1968460d670389e836454abbb05c41537956a937e6e845a2ce37a1d539c9c6543206e9d65759b6bfe90c0e5a0797065f01b8bd15ab65d7af6f0f3053939e6481715409c2c3cf9ff273fb7d49f99b29e17239be3d4340906a0192a31f96dd792f30ebed42fb6124cf9356b6084d122f3eed7fd8cd1ea04997d9cd51c8c61ef0c5f593b3230540d24c52c42ff0678f97ade8f59a5d0cbc083bd3b11a13b7b399e483f73ac2cf98ad26656fe169a18f1d319d1cdd66903071ae80d7be1dc0e1a1a5132a624f4995417b59ad6d71e5a08d8722b43500f675df7623e54495ff8044de03af74b10fa26db4a8098d740ada5aebc43dd43ce0d9d9ee7d17adde7415a52c364143852ad26dc34f4ffe9257ac0a82a90db97037347ab601e97a2372f87b0925629bcddade9c6625c0e9875d3054dba524c581c3785be32992b94dd812cbe8aafc5416136eabacfdb4f311292d67f1acf459f99f0670af5581f3036dceac2322df5a0bc1cce7279912d20f9638421f1b989c1402ad2669bcfa50c77fbabeb40bd28dabbbcb72f1889498569d304269dca8374f61038a3715b38552f92fec649b6795ce692e441bb7e215c5628d4bdbef3e81054be6c7b89e7137cf17ccd45c83dc9710a0dbc50753c3bbaf15f8a90e406790be4db8f528be29f090b5d9908f223658a4a37f03b4fc53fca4119eeb2d1a712c11e7e4b75f41fd59944cae1ab2c6d01a3b9bdee689301ce9cda1cde0dabf97d78bfcb60ff0033c5d9034f9a8f50145752840a949f2ddf255c0735a1f5968b79a3b2436c2e5768f935bb6a58f41be143c9da1a21005c528b0daa3e9c60231deb95ad0fa4d50e1ae4f6f5c508dbd1260318145f6f86d7b828acbea3db71a9a9481b958c600ccdd8f47d3dc1a671cff034c0654d401151715edb38a982e2fc95e9a12b9045a8426969d40a5f94b11afd1ec742c508fc3b6cb783a5a1b530af8176d910e965671b9060b8b15903becb7b1cc94efd5aa8d8fc81bf704e37da1e6727877acc55d12dedc5f830d0b55371b727ee84ddf0999c2ffe695c98d0a5c0712357792fdd80c9796486d23a42ba46681fc667989941014ba48c2e443bd1a176c61733899d65273a" +
            "853028eb5ff34b08b0d4fe233e230288cde6a7b7067591aaf3340f9f964b6b06c73787d9337f926cc4f26632623ca669d73f2269a6a0c336502417e9a4148fee3f884c20cbd8ca700f4b41ded93fd84f7f9fae41fd20a5604b52f018df285deac2bf0cc20c2701848436bc244a3111e07032e2e43724ffc1a959a7a7971dd9891f0cfae55e1ea9448ad8724650f383bf9241011d37cfd7ec298100ab0406924203b0582c832e64ac13b4a69dc528286387cd77688c9230b75f8862cad010bea4725952f716b694f72ffeb9dbfb89ba172f0117792083b44c81cee66d9a7126d99cd89ab5e9e03f0e23550d37a88bee39339da141dfe27f1a9a4ae1c07d0c0123527df19a45effff06b7360b1cbf96aadfe2b28ae46a9b1a8370ae41c5c3615112d61b0040480b57e53bde2646580d9efcc36e54310768c57e266b48965d8dd7c45a433b961949e5b0c85dc40e780b26f53c9ed353a6ee21d9f6a3ff5616c48778cca85484498b9857a0b949a0fb08027f8a0552cb1a3225e17dff2624834453687977f50bd60abf954476c319fc2009f57466eb00e59795143a0b1544bcca22a6a893b438a83d2a80ffb918cd3b2fa7e6fd2a78b03eef8600fa980d92951b75dd20c9792117477f2274d0387588d3b452c76a2cb0477842cfbd4910d7eeb8d8097845239e04b1de49fbdcc09d35e8d47fc447bfc397eefb6f1fdc61c2bbc8e1c655496a2d694a1f8deeb9b3e08331828c9c054b204716c1a5c8504af13cd9bb6379368eaba8cc34d529fbd9f7db177ca46db89cb7259d305c80b205b4337b0514fb60e08d33ceda34c4a4d51429a12b808c46f63ddfa349b177cf8c8f4488464a22940b2f17dbe5f852db4640597ff198f7b5feba22888de7bea2485887ac56f54a68410fbb6ffc1a39f98e9a155059daa88bb46e6c86ec4bd4f049c8807ca10385cd6fccce5a64b84f999b2aaef69c230734e2066b6f8bec19f345e5e7dd1c0ce56e508b7f10f19cb1fb7ba5faff7e152aca87d65d9c6a89d03300bc0723906853d036ab50d371122b5c772c435ea1b20fc8b1ab1f7953afd1d29d3a9c53add9bce2684956a5314379fd3030a0dfa9e81e6da253a43b3a56b7cd9bdf3473714a052e0fc5db75acc502ef409f6cc267669d52576c14462f8dacaec67b6d646eb3eda205d794f2e8ed34dd2ad2501b87cbe07b1dfe0a9705fa5113c024187b996ba758d915b1c1a3aa8ae76d45700b4b91c42029c80cf470f2b1f48d4e0a9e02231eaa0e" +
            "a21a9a0812344a79eb329cde74de5e85bba7730921d75d09ecd70f66215f369bd24dfbe8f539c5fafa31ca7734cd8281c6ea7accf9904581f03d44c3498ad26417ed65273ce8ec7565cb86348b01ebe72e4c4244ce853004dce3a729f59113993ed64ba814e8368f1c16d0bad4df4d2002b660a4776547f6151e4cdfc9fc075993f446ddddfa824c77b5cdd0d9147d77530b1bd917651d6e9b325e88da6e6187ae3b4ce32092063c0b838def17e56b0fe326d1d0e1b56488b90c7e13f2db0d96010071042a2332ea8a423f398d8b04b768eb79ca68cc22f43c335b2dd6148fceb20c367742c00765c417705d3786063c5f23785bc34f48a4ae600e7c7963c53e2982dc889291f0915395b1573015e026dd1ae9ca39dfb9a2d2e73e16a4e408c34bbd99f727725b77cfbf5f2825f95a141a770221909c427d59e9264aeea617d0bf948ac3f3b3d77a590c0b44d76127cbf08a9a2514317813d55572fe7bfc8c27118c9f0536577ec259b866e6c084bc324a5e97df2fafdbfe060ae1dfc2548eb167febb076e3babc0f705a4a1b85e8873c142719eadc4ba9bd10c3015f6a1ac5f349288f43ddf34edecfdfe24522be76307e49b266a5dd3ecd9c45ce9fff1b0b9e3bd1877b7f0baaa8b2051316570fe1deda3f5b52c54116b5b6e64c316f5bc906c43a747e477f3e75cfd0ffd80613e2df98c0203b55a1e071ac870366a3cb217589f8c11b4f5405ae0f9ea6cc6956ec628b30b2b890c300f1bba21e07c63d8a4030cc913fca60318db89812bb8b19c0b78f5d3fd9358bff17502c7fcbc9b9a840390a0e81226a4516ebba7a1928240949a915b34d5af2076fe43528411584bd70112ec13002817e2d2c3c2cf12c2fc29f6b803d04914bf9a6c911a512630e4af0462b5d02d11aaaf9204b92f74e3db0d941d9bc0fa9a9ba0db2a221cda737c7d5edc90a92be700883551f752443a41ebc2c6d92a227e174af751148f88efd74a1a3e9b5113b5b77b9ae185c1a1edb067b2057744c51b2f6b733cf6343fc3f93de314e558fddd52b33935d117d7e77b4053153c4624f853ca65023ab7a68fade2ca474c00431a0aa679ea2c63f93d78b0a0dbbb06037edfa4f6ea1996c075e062c431c21cdf6c4633585e5288af10d6bc1b90c57091713efc143b59eeb631dd326eb5b89d92a63384fe342f49c2b051eb68d88ec5e784a9d241db78c7306c260b0de1740a7cc21ebd0a84b21ec17779beffafc82775503b2cbf7a1e7996925abd5fff3b4" +
            "caecf8332942bd5e75f34a3570a3e02c181b93374883efca147665584783a3e18467c9a27af84a5d48f8a0d9db57158364a0344f5ed4d787b07ab21085f6114ad8bacf76b506759751162aa584518672186877fd0885a8c35525734357b02756504c85b9765ea3f6964a20516b5757fa245c2cfe9f1bc0e9a1ad36309eb243c2229c1fe732ea9bfd36c8e183996ce629eec480cdbbaa4703fe8962c91585e27e9976ea6c620979b6db8e75f2c3f18dc5e09f9e4f9f615ac9274868c192c5040f066f870262f2e41f1510e3327f5aceeffa025024faa5b6cb6b83aa1e4fb87766f94a7586700f9cd635915f91f9fe0540322ee96ab373e2e0c5b8ff2624a387e1d8e20380d4615d4dd0a7cd6568c4ac8fe5b3da2deb69f88e77c52aa0f3225cca620e44c407d2d8feeaf367520529793c4d7f82c497e01dea8f5cdeb6b642b6835dbbf1886badc0a82fdefcb489f0c5233ecfd981f9bda484cd07270885493bd61d651c1c563f9e3f8ba70ba686d5b3dda87b22ab92a9861f386220f76616f37ceebd33246ff7b5127db5ab499ba75d6c756d3e436a49c812cd65b5830ca4932b43a25fcb5c73621c4c6c551b9a6d250af278d2f7ea1a204caf6693b945f34f4a00255523dc150e540d07253cf43a1bafc75f6b72c71697cc11f004badb7f27371d8b1c193e88fbe48675ad43d8a255094f214e867cc34fdf0565635309a0cebfc8122c3f1b9f47175f1e1543174adab2cbc8a89eef229eeeb80d4a83f901d45997b8a72c4aac73e40ad7762aabecf2a229334c3c8977650496f45f56c4fcddb35392dc92c055b9112f2f5b905abf4dc10595c97a080b6cb2dc82e8ce4a79eacd2d75d783a5aeb645cb4ea4a118da0ca1c920d83662b7cd151286be786bd48b93132c464d9561d60595067b805b687f5942c50523b7b6ba938b2bdb138e1906ede55b195b6868cde422b5d838fe1118a79a966a04331dba67a80545af37e5cefd38bdb5076f8a5bd08ae170c5db18e32afebcd9d0261815bdce030c5a0bff4afa360163692998369dd6f5c460679fb27ca7707b4869e42a432fdd961bb1f8075235cdbfffb4d0584cb2c22855b10023f0edc61acabd42665f6c48bca3f04e60f1f39c8c882f3b7973bcc81dbe152927f4d95f9450eb8e5a53ca9827dcc51a24c750fc45ba0b567beaf41eadadd5dc3938330af0bf58f7e5b076719637f70d7a0d1d89a1ddd4f37485bd6b20c3cc58e72696e7c632c5fd1d7ad2ca23791dd2075de18e3d6" +
            "7a6c43601484ca13757262188be6388086caefd186380032380554914eb2739d96acfd66d081158039624b3f3b89249074812727fd1f0eac628f23e55122cf210b50ba8b9098dc5ae9a6f6b439788a57b525c8b7eba6fd56234e9d18657ddc7f36ea8da19ec27c6f0962fbded4fc0ef78833125d92b5cca13a64bb006663f5ad104c0395a75f789d88da80e42b47c6bce43e7c8dcedba326d654d6166ee4ba02537c0e844cf89804c302c688eb4be46bc3986aabf8f35d9d433798aec7398711f9ae414c675adff3ef37267f162f955bd147cadfcade61f4e3911faf615f0191ddab8fc64bcecd972f3bc20755c2b49edefdfd7d9398522696209f67f39566e96b5c960f254f862e1ae88cf327e8064eaf6e55898263986c57351c4632c0f885dc8950c9e1b2e7f63b0e866e6aa6e0e563d693165a782dfa97fed15c57735b3540652bc516f9501fb411be95576a1a443bde64bcbf47212d07f6bd8106e9f97a1ef45fd645c9940eca6fc9bedee9217989b579b4990bf790fb18bbc5f31199c99195b2331f68045d89cc3a554e5c17117913591e70386297d7012d1f61b4ddde16ef7e1b57246591a2f7e3651c5275bfb6dac3ad1a55ad2bf363b84ce93fad8a9cb132a1af2ebfcd3f9f112232e5de8b05268352558bcf950fc8f4868cdb4e894764b60ab0507f5926b0429231c22d3e8dcb49404095500ae61b319daf4e27ac9b691e809d441c98b2f0ab1f9e7f99175b70bb20bff0f8a6dcc7233df848ee723d07ec3b72d157d6d3a2bc2f53914b72a1a8073e96c5cb11493f21ea195f56c0afe29ccaea5b617f0529f357c39b39d96bd31a811c190d6aef0fb222e400071b4f831803c4dea0fb090e829b371663a2417d7d9e4b25e66b06d56c6852684dc16606276841ebe9a340d65a1fb419541e9ea401053146213e2093045f51714018f4015cb398f6a96648425be44abe16d3a6b64db63e0739d4fb65280c0efcdec27e7c0130cb120047528c912b036f5ab18fd6dfa50c4ff114a9bfdec450401f997a470dd0eef58655025f47d45a4b52154082d44ebfc3255e2e81ca4f5aa600cf807ba7b761938cce8fcad74332a48effc8707cc7ebafdd708592d0fcb5005695de71975deb9f24c1847c24f7af58956bbc1d4b1f216ef635cbc78f352e71e3c19fc0483dbe51e4d5073fc2ef3f18c0868417db37386c5d03d5adb28c227f667d26bf86f3de79caf4139c6c4c6b2874e5deeecdf31acf984c4c2fc5e052e88e8515b7c38" +
            "75737544370b273e53c756bffdfec54c40fc82469851399d56432060d02d500996473bedc05fb8028f4ffdf185e1b565bfc0da9325eac3fd7b7375bfc061e0ec02c16e858a987eea418c6db513f06fa002bb938665bab851cc432db2607a17f3c0b4cb28ec856747a1c1f6a1a9645b6a650542345a7988d81f64e26f8e610f79bfb8bf98f24444ac421567ed20015bc5cc889606f559ea3471b085a5fa9f61443e94cf5b87883274102fd19209573244804285a9b3856de75ce42218b216c9e45c3f5fa3e2b95d1b46a44cc6e328269b068137f37ae44b702128eee5d52a8bd870774ad109790238b5db4ef25d1de1a8210889d537e58722688cb5c76c05f1dc67c2ce1748dc626145764c3085a44440626f60f44c3433574c88d3468fa23820983b0c234559e1d87fe4a057f136d79fa1c43dc20569b05c58ec8c507ad7f708b0cb65c00e5a49b2a87535de183ebf693c554006a52e6038c6b0fb8db16494e5c0a365a0bb7b4b123b77b53c7bc88250d91eb3b91a229e1d8baa01bd594261b1220f8600770d91622d59ccf613d7566610fcfb31644462e6abdedc9c7e14b423cff130c21119c7c5b38ed75efd6b00eb0980a77d66ed2eedb3bb1e668eb2ba2813c6b7436d2de03ca319e2e78efaa6fdedcaea4446fc26cd1f6216056e198e30532bd11b6724019a9bd611b3374faca8d630ddf39d54f58e0c1929e7035fdfb4f624fd54b78074ae5a52ae4f0ed7e5e6640d2f4658832871289b84af73b5d6c9e920803b82aa8c0b96bff701ad64a329e1fdf2661ef7dedc0dcae9d7593af2e48ac64818d176df8e9d4d9b3fb3f05627cc9932ca7c9e112b10b5bbda5df5fe6d64a580a454eb77592c0656278d27c8be40ff2bd6235c48157336480f4b69008c89e7a81e67fd146fef949114b75a806f899815b10c75ea7caa095e1188d18348507586b96486095f0ead66a67b1d33868714235ab42abeb4e903b0330d352d1870728e154d8433f03a97b0bcf411ea74541a31b7bd9a439251f74a57fb00f98a1ecf1d7a0f114707e352e96b81e901cfe5f33d190aee198059d500b435a7a2cc0ab54af6f71f3ce208969c480a9adffe4c71439670ee779b6dd963c9e5092560162994b55741cd361a71e95d449110be7ca7155d5b2fb5642eab14d78e26dfb74650d962010430660efbd72a6f30c95debc9ef54f073265b07531873acd0407f62f021fc621e7c2ff42e9be454b313c20ff7be10d30256fb59ea3a00c2fc9714cc9b60b" +
            "a45c635836de720bce6212104e71950521d761306178941061cb1ae02e0107f78aa9a41618dc2c89b24a6057f4aed38d6a960b8e9093cbd63aeb5a11d36dcec93d965f978c42e3d874e267b883ec5c508836f4ef982e4914c72d1a7e92e0707e1e9068aa8c3f303d102696ea006002fa2b671f1cf4adddf0aef381a526ec4be4272fc1119c2e00c2d7222fcdca41e0dfda7e57de86c87e3e7a43b4337fc43ebfcd20b7aa3e323ec93f05bb84df02e4ed67b92fc8531023aedba11790e47cb8b191ba026d3f25366235cf24965f08b7a0c2b65b1be99c894ea35b931fa39bca7eebe96857e22f77562551187acaa450347171579d705f681b76f39c8243140c59ff3d26fab68eda2acd4d44b38322291d31bb32f8edf0adf40bab0cbd8848bc6a6c0ad7fe31128a33f434dc82c675c1439e08f2a18bd18ec3d0c7b3de0cd3880bb98cb6a2f9ec99bc1a8ba0ac19218567481589d16ad4ca2b5664ae3b54376569258d76291f12dbf3a74f28e890618f929dc9a37f9e7913741ebfb024c7cc85061e9a731b6d770386bce02ae10f6df9ac23c0d174a9c766823b825c91252f631070be7e4a799fb608d21928dae8462b2fe6a5bbcc82a95ba5f42f8dded113fb4088b8c076767b8aac5cfd7f243b44015be98831d8ba531075cf1ad712cc607fc1ace10de39df2fe8d83710e8c7990710baee775386857141f289e16880cf9910ad5ecfbbef854055404503caedf237489b92b47ae5071aa30a5556a38855aeb58860a7f517e6c7c63a53ab513f6b8c292743a5cbe1053a615b5f031311792be734cbfb611303626265847d6c54bab005b09280ffaa715b58d5592847b990a2111fd57e07597afafcbc6b82ce2d5c4220ff70db53fcaa44ffee4a20726515d8a2211f2036319f685a8d844bfbe25f810b5ee2ce504951229a4a453c821e2de8fc83f6feb6b614e4ebffde6cbc8485c3ac9ec46221fcf09cd71aead885a07a60b97bf9efe65c2d18fe7beb7834651939d463b1d63e4c1d28ecd4d3cc777de5144bc97cb3cc5ff10824a79d9a6734b1deaa9976760edeb20105a14fb7900a864e1a8c5bd57c1ceed327ed813baa321809bc050eaf40a6a0e2fe95d04c5a0f98545fa10715cba741267f4067bf291ac35462e9534f7f16e3fd0e70578073994cd2826ef22cd2dd5e1985c5236bff6bcf2cdca7c21da3462b1d4d65185696e1c34457610775dc4f0625612db915bc06302af1e79c30dd988ceb9a99ff08abffd579baa4f87718" +
            "3a9986d21b10c5815ab33132f53e61858507194766867f020ed30d8f8710d065f160ccd1db901b04046a8746f52555de5fcb680c08defa8e177094cd7d82f33576dd1b237a5662e10e16ed087032ca31ac4bc6d9585f3d388c61bc2875d134d505872cefa2c39f70118323b984b4cc5b56d9cc8368104bcc47f550811c8754cc1967357fbc1314f98e17b415281213b35c7af687be267432f06dcedc6914060d4489daa38e2ea3171aa57e5b5b1ac42acb8420ecc776093a00b1e68b49f93ce632965714aa5a4409f1357ca43e3e84c1e40f92a3eccbec2edc14b0374979258182696abed13d65cde24b1f9255d406e914e02977cf6e227114af6729c73df91c88bf544fd50e2f9ec2f4d17bc458b780556b6dee9a481b3cdd47798cc5387fcb5fb43ac5503f176c608671e09087d44977e6e87834a40b2838b459dff89fd8e001c9199df040c09b60afe2e60900de7deb6455b07cf22caa61394504edcbf2145f2ffaca5ce1a032f453390567a5104da1893cbce7e6509cfdd53cc25066ac6dec5285ce1eeeda5558942eb62e001bd274264d3c785362c34c6b6eafa44a418c6cafe70338de0580783a5f070bd8b0cdf26b68dd002a050502e427e9a29b5ed155e06fabc7a8960862ee6bb04948cd6ba47f3bc3e3bbde4e65b227efa7696dd7dc068acbe1fe210ce9eb4cd79f0ac01d8e0b48e89772f2c30506b2087ade6d186fb4067069e6a90dde74ca3f85f1744066c913991ea434b377738189daa7ff5b7b2ffad933f2d4d083b413dcfb93ac95f6e3e1cc01d06ac50dca6df88bdcb61ca8eddcd2fe9a3e9a28814ac7d9c3b23251751b38c427ce8b978602f1a20cd32ec7422c17a8ab60d005b78cc3326747ff20c038bbe794d0d86d049ba729573a03e0d0bdc87da0b51ec7e91dcbf585cd9c78494097ff11d13325ae8603a90ac7a02d3fe1568fe71a1fa738fc8590353af554579640a9a6bfd298e96c97c87a7df2b97e00d97e8a8a8af7f61abc070ad7716870dec683aa574c56d599afd8bb187dd2d6f4610abc4b9ae34b8e2172025bbd6a41f9707c5d89ed7cfe5dd3069a5d4f8a54e3a748ee0b20ea85b3c5d9bbff277bb1cba6b302e01e09921a1c00a435829f2cc6283454c655bb60f50e1925fb918859fd0e298444fdc70f50154dd8a2593ee385c03972222a6712d8377377f470216ae30a387f8e3529474b94905c65a06a2733915ae55fa188cb0bd0301ba3fcc72f468569a16b04916e2f31746ee72b985de1c" +
            "9e19cabb06ca46f1de7765b10146aeb60dd107f7922243999f02505198fa91f6f9e92f0a0f8c18a8c67c2d6bde70361420f97525a74fefbac297a9e08780060084c101317894ec247722484e488f78badcd909b08a322b734e1509f406a3616f25b7dedfc351da67fc29bd9f661f86728153789c2b47b52fd5a5ee3e0d7589cdf681eab6b1190bb89695b3a10c0e8988d0580fa39a080e036a4efe1be4e195313109ac2d29d2db3973abed753e6ee6198914a119298bc19ae6438108aaf3c61c7a96a2b7276aa3f32161f33661acf0cdd5c21a562cd83245840796c50436fa8c3acf59ac874ee0ac60a25e284cf04275016b391e2172f418ca0dba39793deda4a6050a3cf0dad3b477aa45bf34746bcda9c4835a5cee6799d0c53dd6af05fa8765168d7d179a55a0548d36bf5635bda512ae6befe0ec14d952a95af6cdbcf6465c03cf180ae7c3114bfcf0279d957f60ab089a4239131d6bd535ee60b909e71a30dd0119e94967eb89f592037ac1a3bf281745fec0e27219bcb523b29412672fb0e0c306ec339326e13d5d8dc5e3cac98ea0d8e458a8994371dfa4f6fb392a57f24a4dd30db0b1d743da747122e56bd5c93b707fe6ef95e424335fee87f165e8459a3e38886e334de295a37f2e22730b4f78bb69ad79b9c9f4f3e7c252c9f85e0292079420308b5f58c21bb048b10f27a6989ce8e1dea7eecfa46767078eb18532fd6b8872e27c7cbd1d467b4268077aad1e762ab4c81e15f5c1eec44d961f47dc3851c2d3ac60f5eba379f3293e00bbaa4544e9a084077c2d397532eb74407bd1d1df847c01a355759d26e24dd981d89a4d459fdb762bd50c00973759b3b078b9c1fa35f053b3c6152116e8c9f469dfdcdd5ce0d171d7cc0696ea31a7ee71963d41d189f93364f996306cc99248fa3229626fc25051cad98be834eb4922c6009a83699cded08180e7fc47f83b3c6acc6edf581a966f8a5cec53f7dff430b090a68bf315c300bd5d8c76a8a39c4cf3bac58129d4b35b8872e3b8de12723dd1d9111a8b2d1400973d248a104c8953fd9eb5ec74d13def2a4201a9ea594f308c8aeeb745bd422a56a7010d427d2e0c484764481717dc04af742c10668191396a919a02c79ea1a225e2d346a5b7190259b948e3da43a31ae3035c9a3a02067b7a341f64047ef22d374741e2f7ff3dcb5d79a991d9eb2408bad0536e96a3a0955bfad65824fa1d8f47e17d1d27b79e1d4b2322b49781eb0d8286f5a084bb24d45778b33bfdb" +
            "a317021e7bdc4007b9a95d44f5584bea9d881299c41fcb14afaa49199773fbcc4d943627d2b704713c4c19d787bd42c5edee3ee4e99911baaef85a01b530db7e7148db012d6875d230fe5ac63f8108f496a6dc637275a87b965046b724a3138204f95723f54679d3f755e67968d34a45bde27ef8a94fdb9fc669b4f54ededd4b5736c1ce0bdd2dacc8a530765ff9fa547b18e844037a3c7b23cd2081cf5cf3ff1ed2b21921ba24f46b49d11eaa1a4bd9fdc6a5b7a57c6814b8a5fbddd4527edddd37ede32493ee652df4a9532b5304dc24eeb5cc5abf5b450cf67e19fc7a935e9813b6bba12d191a32067416fd12bf82b8197dcab44296e344575d5c4aecb7d84093f453311c213bdf9357de2924fdfc58481c134e87f7e55b8bf8ea8eda5739e641cafabd38613dc4e74dd78caed72cea43ee53a12d0c6d76fed8822b32ae34260417372dba4566f095c40d34d91c2a05246fd6ed22972385da8a0bf1b1f074a0c99437844878a45fcd2dbbd6a187c750203fa860c1a967beb9f0191b88ba90c01b66aead6118a0264498b54bb71f366f33c667d0a1332d4da1f6d03dda1299ddde1366c828456b40ccb1bc9ac012d1e9805ec4195dc3a3e37487bf5c4dc341d42d6afc7887998a2a91a7507901f4a498d202adcd601574906baaac4447a34300e50f6ee276cd0ab6ddb03bcb4eac0ebd2301a454192fa61f36e8f7563cfde81ae54375b7f22dc458cb3384bb3cb1f79cb0ce67543ad34fef029c8c5637c499569baeb96625dcf7cf3aa21a3c64c34da380ce776311afac8b25696d3af113031b8e773dfa607883d27a93a3128f91d21139fa24697c2c309dadec570b6c844727389dc42690672c4da61a832626aac34f03039bfe455001ed4131a5e8742c3e689747f1741d2acd360f560edd7cc5d90185f8536c5221be675e7f2fb8b401845fa67897cb07ab7fa47ca0dfcbf32f38a3c3ea18bcd2bb2f9f526e625feea0a7d7f6cc6543eff03b4ae13d0ca4eaa2bf3b63c59cb26cd79955c62525eedea7a505c148fa8ec39e03b1d7d7b5510ba2bc120b1368aa6d322d16a047e09b6d2dcbd1e13a7e1a020bea339ac745ce19979163f3c949087564dcd05bb7abebe6db7dfdef67f767e16f30b7e3c4291975558ddb77d3dbbe9becb61d32414d92f814ea6575e3c0ec73341212061ef9e5768790605138e18b92785ee4c0b49c3aba0d2aec38c165744f1110a3fd9da386d7b768491d0898a797112bd3c327c07909daab4bf07c2" +
            "e59b14e91c09e6872df4714d1b2cec69955e74977095b56dd6b04f54a722e095df4345b24a0e68f8276c9ca38dbea0fa33604491fd63d2e705532e0a1376584f22e553d770d08de8777d33e6d90785fc9c52d8157a3f30b6143d7566104056a4f9d83d0273ec436c99478286f0726923fdc44cb3eef4841e519ebb3113887addefff2f4cf5203e4faefd59feeedae1b46358ab31e53e9c659d681ea0ec5436bf7c31ad92d3d55ed2f5568210efa2d0eb8c8e18eac07676f42dafa1b3644ad3d1c6499a008232ed20043620920cd77b89b768b629076bede954b8baf1a27f5c86e6681daf02b516516eb3fc9ba1dbfb54987d8bf57c1e7c5fabce24f896093cd29a79c14d5490d98ce6cbe56eaa90eb87124adda2de00d3cb1519b331e7bef7e3f277474cc9892cbf0b4d5b4d8fc842cd109ef0a869f2a0ebe299e16007ca40d3084392e79f1694a19f75947209ea99a9467578bff1cbd1a431f7811ff8b3f313e34ff7e394eaa19073c6bc2e3a1bbff638fea5f4c3585a3c2c55556e7c4bd476197ac85d8388a3a9fa237522d5c064c9feb83eb9e41bd2ab9a0d23eddf40afedc7ea3580dd03dde9bdcbd7ff69fe3af039db9e62d80eb3c2cffd1277cf066489ea12b39ff51beba8e41608be3ae28e861ff408c2d28a4965c1b60440dde5c0e2b667cf9b21e40052ddf1fcf8e8799dcfb2bd0bfa16ccab599b28f6701043818754a1d10a572c8349187498ff7b890fe4fa9bfd0463b2d7c8154849831c2c0547cd79e946e45fcc651731c48980fd7ee3e184b54be3a1bfcdd8424bee71cc61ad2cd5935cd11c103a95422192b263eb0f575c418faac2e5c756b8f7fe0e45ead8d22737c14458094b0f43c2127925f774a59cfd06b1332637fde0b24afabccfd409eaa86f29bdc4c1127ef6b905be150d00360585e28b8c3d1c60c0b18fca6ff1730cff35c582e851704a11c5abaa94ee5776c6c905e6fea6f0d7ccd7ba5037baf6d68b05d1c2f6fd9adc972d86f03711c76f0a7f0ad26ebc067abc1ef0fc75bc64a79a1917f9c597e4cc2c46d686e59bf71029fcd613ba5afc1a34456f922a439d9f2666d78a212da366c7746cc97da6fc6acfb7df1a7a059d26c225b5437fa70f6e9f0a911d99ed0c44335efd5bab3c4d6d5c3383af4c1b63f581571748fd0cbc9c0fca1a94da3892907597db283631f82d295504540fc7e92d42fbf8ca8013f118dd2956d96429b504a0ab291cf5b06ca4025071a976d7c3cc69a6c53f2c180da99a3" +
            "bc73f76818f499728fb66221a01d1a053fc30ee3866b37510c27702efb203e105ac20a489b727621489eebe04e15827ca51fac5900b9e603f31e198e4076620118d8cbdf4be7dde4f2291268a3c61749f2ac9ce2a8f9db749ff507cb77698d09e2b75889b7d2f81b5ac40bccfc33ad23eb18dbe4ef1b39712d747b6dd8753685462bed1af78deca9db7cc6710a571138a100d848e32471e1a11ba4e9a94e07bac0aae6367722d733dc4130ec14866757ec711c388ba51dc0775e80d7d0b2fac240a12c176d2f64bb22fcc1cf912439db6d4d25cbbdcf58f13187352e62015e88844fee42edb75a54b6bac667fdba1fb60d37cfc356c5b27187fec87052a9d9ba7c6a5ef9bc64d3f6b3ed2b8d47cdd03f4822b6b01082b2523ef7fdd0e601d121998d7963e4b28f0790b1864d8edd54086d251b21f2bce2e75d14925db0924ffc2ff3090968b8c926d909fb158267c46b80952d9668bef5d45e799be34b47d330be52dff25b8afda5b1a511a6c675a4aee30e17b5cb0bea7ae5bdc073fe8dc739dde7bfe0b05dc94152dbbea94c6e25e310c39cf99cbd658b34f7524c55a4ec36e15d4e2347718ddfc8f7b958f1c20ac38881d2abb1294c378fa7669f1ade6150265845c3120192788a1ef6bc7b88d6a923c14623f2889e2042bcefe5e9fe992938233db0d0e94724239611072d54e689ca7c29d01c4fe2be7489668bd2cb8e153948a2328af704ca5f2b8c225b431d585e2afeff25985e57bfebdb2fc937702ef534b0609062062ca782bbfdda99b95b6331e30451bad81deccd3306a5057d939317996cbd0140cf3f1f3518287c4c008718eea43b2a1095e7433edf96584d1cea0af50a1a393ef61a34b59a8cbf33e04c23e560ef7f1e4044ca353ab28210f2ec7f6b2d283e5aeb355f44de2f49a96620ddd573bc5792c29ca9ba4d6e80d207163182ab89f7f346123069465706948802b4089676a5fd6e9fe3d593fd89c9abddca5625e9a4e1cebadd548e3bd438850641801d95a46efe44f1aef05fde299396b36890bda71191b566028f81d1d692e52638f7f71f44e96e704d9ba436b5322f7b730582274dac8b9d9d126b48f24aec14136b0f68f997153ac62bdc5da3fd578cc1a6f488508d35c77a68eb3a35fc08bd2c31ac01a1fb755a446da9c0360a1326f5c6e35370324be80008197836451a2ff6395902f70fcd06cc762050d40f9bad2dc7bce97dd9c58558a7d53c2420bbfa9c1bbd040d0f254247fb2f22be98fa40cf9" +
            "45f2c989fd53c61f4e39902f3f7d55e84158b70e3796774349f1e54db0865d1243645aa0e2360217c2fb45cd2fc9ebb338123e7415b1795adee99b5a1b62f357d97f4157d6bcc04ddc417cfbf8f5d2a0667d2f673d7c4d4e77a10c80f421a4dca86617b18a1093923b26974cb8c9621a4c51bb61bb6e5587aabab799dff6973213e30186c5a915a3112d3408f1210e5248c38b88bf5d8985bb087211b0efb1ac435258ca0b6e4d4d9c595dc8ca6af5e30ad2eef9641fa5f6dd10fe2d87df5b6c25983a6e177c5ca60569400d9d019e2d406a698a9634c96faa35e18263cbca57fde515f6f0a008e1ff43d66c9609db2c4cd7d5f90a9dabc379cfce3f170f762a97accd51a8d95295c6691476be6fc3164563884531f94af6f823544fb03d909939412cd68dd58ab950004f6e291052f2897f317a25ec091cb41e1a982d24795bcc41dd9b22796fff2d98efe37f78346fdb8502b71e44f906f6e1703fd587caed53ff945e99e1b1b30f77ba47f5f5c583f50e88a02162c07d5d96a63653f5b6b48c73776a0112f1645a723f21b4bba759ff07e0aa0cced171f2ca3e29bbd30f76bfa41cc18e073d998a3a81942a5ae38bbb38465420e6ebff920e41b85fe1f8ff1ee894086080829f98ec26f24c3f1d119f9fa0c09570c8761624e1fef528df96e6b0945d26449659923f202eac85f6637e78e23ecaab5b14ca4f47bb9d3f53e76fc3bd2b96abb0e69ab2ee2c9e891b785eb99c2fe66beba15398d2a110169818672cf8f3257b80befab476ae8af3caf07f19a0f243070b7aa7c8c1bf7b441ca6404346300f65168cce79101d4b9f8196bd1db0e1cb7230b3d2f3d6840515e0998b48035b3d3210354eb0482da03392ed4a0154447e58b42e61206587d4c42b44341a69d0b2193d41d41c697772b278c8f3caa48cc18e361f927707a3a529a4804df5d3f545b3e659dfb14df04e70a2a323955383292b370476a4bbe27b4cd757b3247dad56bda009e3db9b8087347919b4edb89765b1882cb971bb2b2a64f741225d83c03de90b27f6e4fde6ce0a5bc19876a440065b985b70373fc87419957680aa980b35a8961a98a474f9bedf46c576987e0f6e577540642e796d057dea04addbad24290dc6e28f508f398bad482f6efa74b0b6f7d7f3fe90a85c31efb590e09a228084b5317c179d10411409e4d8a9c3972a2ab89195cda0e04442c1d40663dcb13ad0ed6bff2602bf3d89f2936d66bdf0dfd7e404d6ee81b6b03f3ec79cfadf284" +
            "3347ea36e9cdaafdae9d1d33d3abb7703b2baa89ec21e82f292e9c0825b81d4372f8dfb10fbdadd1a6fd8cf932d08447d18af564312e52c10ef76697f4b61e8451edb8f93235b2333b0f17df9b30ddadae48fa55dcf5e22824b3c423f384d585d3274db1651194a755edaf0e53e181e6426b6c0bf811a6fc981b61d77252ce330b6453d91a09b4330bae7cd791441df6a6eef5f01412fd349fa044bffc64371bbc99f1310323ec4fe880c6b2c912780c2cc99584dc7dab48d6a8e3a696d161d76d2540c9ad7af9d7d248aea91f2b54315aba721b4ed14e9449311cbbe94c0ddb628269bb8ff15d1fb654b68b10a4247da6df3efa02bab43f0e0a2a8ccaaaa6a6a9455af0924171b090245ddd5364678fcd4cda8bcb9d08cda3c71c15fc3be4ed0a05d04f1d078362aeb301d36ebc0d60c8eb4f391c16cd7b709ecb6c5fe8bd7438917d338f472c4dfc15939e423a7242bb1a938c5852194cdf6d9ef917b15fac3bdc8cb4c44c33a8e559895c8c772108b04106890ac76b3ac5b4281a9166e4ec2ef35aa3c13c73477cad53c0cb3858a63afdd2e2c0b533866d48025c10c0094880e5122022cbebf4d4ea4fe2896d3b7bab41f5499aebdd255cfb56b08a342515822b543d34255b24cbb86b8b692e71b98869d1764cd1535c55f7ec45aae8609aabb5db8c3473709b3476a31bc996c4f749068b9edc892e214c3e24e9942beaea76efd33fdc8c32b181fa751b4a2ff0bcaee08b46d5461d1e697bec33e31f6cc79e4cc74972afdb393023bf2b6d7cd795c76c9927ff4b3a32824854c9d1c71be9357d9a7c1219a308fbb165f706d843f42450d36fbd598ecbf08a2f6358d0f098d7862d5e6e864f88e2ca68a26dffdaeed3cae33b39ce40a7433b5b77e0febfde4c637cc771d0055dfa65cded022c91bf7ae79b13ab4f9ab26f41014f56529b61594620e42b1507fdb23f6e7c16d4b140bdd2a1102eeb8041ca1146995f6876f58447d776589112f70ee85f5511e4f401eb6f3a05733b2e61f65a2abdd4cab99081791464735aaa330884070f97e3727148fd27e4e5f7fccfe01d08bc314c2d5b7a859a02bee46f3a175e5905f4fcf3f556586444509eb49315d41664e1f19797e4c75579f506649ecccff55735bfe2e36abf7042084a7b14bfb15d3d270788d2b8608f1220a685c1b53cd2f1685b58720f0611b6d4844ae87514e32d7cd3bb3cbbf523696c33ff15808c8d615e57d0cf7128ae590e50c2a508dd00fbf547ec7ca805243" +
            "db92b2c9aeec46f883e79deae94a4dd21fd8a6d73468acd9cc3f34f8f9e7a0d91719c23a7f5eaa7e7dd481293a222a79fd2e4a0f46905cb0c7bb93f226a5894b7b275875a715b5a0ecd7c3d490635f377fe45a3a35564d5c4f9022c87de945084ea4fd3e59065afe1732612884e88dd2ed84ab1e86719c1131579b1108acea49f8e9b3bc4ec848e694402f7ba5ebc99ead07a81c0c2a2605ff1df4fc378483df91e49ab7a8efb0aaae709fc188284dce5e9998aa2e0e27fa3c746fa6e2e8c91a970afac584c14a334aead5fd43a4be968ac5671af27330e20d0f2925218c2e6c335ebd3759c1a4815012dbdd0c8d48de6fc52c64216f7363e8391785fc815dfc92aec2ce7d591566f171e6ecf48670adc7eb0d68e58a7624951e062b60b89e6e17161a90dbeefa99345cd3a0ae72cb4aba87f697c137f45252429d5692a4c2b2a83028d9688fc357fab0514bcb8aad120f436112ece6d214a87d2e35367feab7058834f6d45f2156e2aefc2debcaa35f38e0fa0fe6eb2670bddbe472923719a4e9a1431af22664b26521332ad501a34e1750bafd7b1762686001c4bee3570ae692ad1d008160dfb37a5f4084978930815e1fc711d16c808bbb69d9090bec2ba8f2c30403989c8bd17486e96cc88ac0990663fc0277debf7cd1ce34de336977f0a13e777651115fcac1e1278338b6563625696f956938d2fcc1f12c4bf06c85c2cd7e5ee441b92ed8a042f11c713985c38f7e4514d12e3db65e3dd7b83371f01b7bd0f6d5ed20893e6f16b957bc931818af0ba55bba35550e7620de325005d68207004f00ca9568953c6a8746cab926e7ece903dbaa9de940cc54830338f203b461aa8ecb38d35b4d7726ea99580b51c1b72e233e59fea2517ac5d6161cd720954d90c5b6d567b7af060a7db66d61384684545e7e85fb9c0944c9d8a10db5c036a13c954a4bf7546b95b39e0233ccafa1ea1c7c0c9b19dd6ac762d6060fd5c22e4fb6a110a0743cf84bc004317e2131f2a30657fcb055c836ece838e71961f7f79a3c5fcd44db5c3bd2392e796531d08dbbee86dd60d382e50ad815fd8eb8f5420e9841645c7f73972c1b7a47ab685f9308db5e9d85845f19efed48ac813169faa911e8da9cc04ec656e668b72e228b4c9ffb9b86f3f080b33aa01e3a1baf696877a15cc54fd79eb2cd67fdeb9fff0c324cc99ecd2d7152a6ac1c4f6aa0b3e9ebf98b0d7c3fdf212f510a37779f51002b9b8875647caf3844d27972d9fdb2b79d61e6776" +
            "47d06b084ad62a22aa33");
        assertTrue(Arrays.areEqual(expected, s));

        sig = Signature.getInstance("SPHINCSPlus", "BCPQC");

        sig.initVerify(kp.getPublic());

        sig.update(msg, 0, msg.length);

        assertTrue(sig.verify(s));
    }

    public void testSphincsRandomSigSHAKE()
        throws Exception
    {
        KeyPairGenerator kpg = KeyPairGenerator.getInstance("SPHINCSPlus", "BCPQC");

        kpg.initialize(SPHINCSPlusParameterSpec.shake256_256f, new SecureRandom());

        KeyPair kp = kpg.generateKeyPair();

        Signature sig = Signature.getInstance("SPHINCSPlus", "BCPQC");

        sig.initSign(kp.getPrivate(), new SecureRandom());

        sig.update(msg, 0, msg.length);

        byte[] s = sig.sign();

        sig = Signature.getInstance("SPHINCSPlus", "BCPQC");

        sig.initVerify(kp.getPublic());

        sig.update(msg, 0, msg.length);

        assertTrue(sig.verify(s));
    }

    private static class RiggedRandom
        extends SecureRandom
    {
        public void nextBytes(byte[] bytes)
        {
            for (int i = 0; i != bytes.length; i++)
            {
                bytes[i] = (byte)(i & 0xff);
            }
        }
    }
}

