/* 
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent2.portlet.proxy;

import java.net.URI;

import javax.portlet.PortletConfig;
import javax.portlet.PortletRequest;

import org.apache.commons.lang.StringUtils;
import org.apache.portals.applications.webcontent2.proxy.ProxyContext;
import org.apache.portals.applications.webcontent2.proxy.ProxyMapping;
import org.apache.portals.applications.webcontent2.proxy.impl.AbstractProxyMapping;

/**
 * Default {@link ProxyMapping} implementation for a generic reverse proxy portlet.
 * <p>
 * Because a generic reverse proxy portlet may serve any remote content,
 * this allows matching from any local path info and any remote content URI.
 * And this resolves the remote target URI by reading {@link GenericReverseProxyPortlet.REMOTE_URI_PARAM_NAME}
 * request parameter value.
 * </p>
 */
public class PortletAnyProxyMapping extends AbstractProxyMapping
{

    /**
     * Underlying {@link PortletConfig}.
     */
    private PortletConfig portletConfig;

    /**
     * Constructs a {@link PortletAnyProxyMapping} with the given <code>portletConfig</code>.
     * @param portletConfig
     */
    public PortletAnyProxyMapping(PortletConfig portletConfig)
    {
        this.portletConfig = portletConfig;
    }

    /**
     * {@inheritDoc}
     * <p>
     * Always returns true.
     * </p>
     */
    public boolean matchesLocal(String localPath)
    {
        return true;
    }

    /**
     * {@inheritDoc}
     * <p>
     * Returns a resolved remote URI by reading {@link GenericReverseProxyPortlet.REMOTE_URI_PARAM_NAME} request parameter value.
     * </p>
     */
    public String resolveRemoteFromLocal(String localPath)
    {
        return getCurrentRemoteURI();
    }

    /**
     * {@inheritDoc}
     * <p>
     * Always returns true.
     * </p>
     */
    public boolean matchesRemote(URI remoteURI)
    {
        return true;
    }

    /**
     * {@inheritDoc}
     * <p>
     * Always returns <code>/</code> because no local path info can be resolved in portlet environment.
     * </p>
     */
    public String resolveLocalFromRemote(URI remoteURI)
    {
        return "/";
    }

    /**
     * Returns the underlying {@link PortletConfig}.
     * @return
     */
    protected PortletConfig getPortletConfig()
    {
        return portletConfig;
    }

    /**
     * Returns the current remote target URI by reading {@link GenericReverseProxyPortlet.REMOTE_URI_PARAM_NAME}
     * request parameter value.
     * <p>
     * It reads the request parameter first. If not found, then it reads the portlet preferences.
     * If nothing found, then it reads the portlet init parameter finally.
     * </p>
     * @return
     */
    protected String getCurrentRemoteURI()
    {
        ProxyContext proxyContext = ProxyContext.getCurrentProxyContext();
        PortletRequestContext prc = (PortletRequestContext) proxyContext.getRequestContext();
        PortletRequest request = prc.getPortletRequest();
        String remoteURI = request.getParameter(GenericReverseProxyPortlet.REMOTE_URI_PARAM_NAME);

        if (StringUtils.isBlank(remoteURI))
        {
            remoteURI = request.getPreferences().getValue(GenericReverseProxyPortlet.REMOTE_URI_PARAM_NAME, null);
        }

        if (StringUtils.isBlank(remoteURI))
        {
            remoteURI = getPortletConfig().getInitParameter(GenericReverseProxyPortlet.REMOTE_URI_PARAM_NAME);
        }

        return remoteURI;
    }
}
