/* 
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent2.portlet.proxy;

import java.io.IOException;

import javax.portlet.ActionRequest;
import javax.portlet.ActionResponse;
import javax.portlet.PortletException;
import javax.portlet.PortletRequest;
import javax.portlet.PortletResponse;
import javax.portlet.RenderRequest;
import javax.portlet.RenderResponse;
import javax.portlet.ResourceRequest;
import javax.portlet.ResourceResponse;

import org.apache.commons.lang.StringUtils;
import org.apache.portals.applications.webcontent2.proxy.ProxyContext;
import org.apache.portals.applications.webcontent2.proxy.ReverseProxyException;
import org.apache.portals.applications.webcontent2.proxy.ReverseProxyService;
import org.apache.portals.bridges.velocity.GenericVelocityPortlet;

/**
 * Generic reverse proxy portlet invoking {@link ReverseProxyService} injected.
 */
public class GenericReverseProxyPortlet extends GenericVelocityPortlet
{

    /**
     * Name of the portlet request parameter pointing at the target remote URI.
     */
    public static final String REMOTE_URI_PARAM_NAME = "remote.uri";

    /**
     * Internal {@link ReverseProxyService} to invoke while processing reverse proxy requests.
     */
    private ReverseProxyService proxyService;

    /**
     * Zero-argument default constructor.
     */
    public GenericReverseProxyPortlet()
    {
        super();
    }

    /**
     * Returns the internal {@link ReverseProxyService} instance.
     * @return
     */
    public ReverseProxyService getProxyService()
    {
        return proxyService;
    }

    /**
     * Sets the internal {@link ReverseProxyService} instance.
     * @param proxyService
     */
    public void setProxyService(ReverseProxyService proxyService)
    {
        this.proxyService = proxyService;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void doView(RenderRequest request, RenderResponse response) throws PortletException, IOException
    {
        response.setContentType("text/html");
        invokeProxyService(request, response);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void serveResource(ResourceRequest request, ResourceResponse response) throws PortletException, IOException
    {
        invokeProxyService(request, response);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void processAction(ActionRequest request, ActionResponse response) throws PortletException, IOException
    {
        String param = StringUtils.trim(request.getParameter(REMOTE_URI_PARAM_NAME));

        if (StringUtils.isNotEmpty(param))
        {
            response.setRenderParameter(REMOTE_URI_PARAM_NAME, param);
        }
    }

    /**
     * Invokes the internal {@link ReverseProxyService} on portlet requests.
     */
    protected void invokeProxyService(PortletRequest request, PortletResponse response) throws PortletException, IOException
    {
        try
        {
            PortletRequestContext requestContext = new PortletRequestContext(request, response);
            ProxyContext proxyContext = new ProxyContext(requestContext);
            ProxyContext.setCurrentProxyContext(proxyContext);
            getProxyService().invoke(proxyContext);
        }
        catch (ReverseProxyException e)
        {
            if (e.getStatusCode() > 0)
            {
                throw new PortletException("Response proxy processing exception status: " + e.getStatusCode(), e);
            }
            else
            {
                throw new PortletException(e);
            }
        }
        catch (IOException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new PortletException(e);
        }
        finally
        {
            ProxyContext.removeCurrentProxyContext();
        }
    }
}
