/* 
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent2.portlet;

import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.io.Reader;
import java.io.StringWriter;
import java.io.UnsupportedEncodingException;
import java.io.Writer;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.portlet.ActionRequest;
import javax.portlet.ActionResponse;
import javax.portlet.PortletConfig;
import javax.portlet.PortletException;
import javax.portlet.PortletMode;
import javax.portlet.PortletRequest;
import javax.portlet.PortletURL;
import javax.portlet.RenderRequest;
import javax.portlet.RenderResponse;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.math.NumberUtils;
import org.apache.http.Consts;
import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpHost;
import org.apache.http.NameValuePair;
import org.apache.http.client.CookieStore;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.cookie.Cookie;
import org.apache.http.entity.ContentType;
import org.apache.http.impl.client.BasicCookieStore;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.client.LaxRedirectStrategy;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.util.EntityUtils;
import org.apache.portals.applications.webcontent2.portlet.history.WebContentPage;
import org.apache.portals.applications.webcontent2.portlet.history.WebContentPageHistory;
import org.apache.portals.applications.webcontent2.portlet.rewriter.MappingRewriterController;
import org.apache.portals.applications.webcontent2.portlet.rewriter.Rewriter;
import org.apache.portals.applications.webcontent2.portlet.rewriter.RewriterController;
import org.apache.portals.applications.webcontent2.portlet.rewriter.RewriterException;
import org.apache.portals.applications.webcontent2.portlet.rewriter.RulesetRewriter;
import org.apache.portals.applications.webcontent2.portlet.rewriter.WebContentRewriter;
import org.apache.portals.applications.webcontent2.portlet.rewriter.html.neko.NekoParserAdaptor;
import org.apache.portals.applications.webcontent2.portlet.rewriter.rules.Ruleset;
import org.apache.portals.applications.webcontent2.portlet.rewriter.xml.SaxParserAdaptor;
import org.apache.portals.bridges.velocity.GenericVelocityPortlet;
import org.apache.portals.messaging.PortletMessaging;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * WebContentPortlet * WebContentPortlet Allows navigation inside the portlet
 * and caches the latest URL
 * 
 * TODO: Preferences, cache stream instead of URL *
 * 
 * @author <a href="mailto:rogerrutr@apache.org">Roger Ruttimann </a>
 * @version $Id: WebContentPortlet.java 891414 2009-12-16 20:19:02Z rwatler $
 */

public class WebContentPortlet extends GenericVelocityPortlet
{
    /**
     * Configuration constants.
     */
    public static final String VIEW_SOURCE_PARAM = "viewSource";
    public static final String EDIT_SOURCE_PARAM = "editSource";

    // ...browser action buttons
    public static final String BROWSER_ACTION_PARAM = "wcBrowserAction"; 
    public static final String BROWSER_ACTION_PREVIOUS_PAGE = "previousPage"; 
    public static final String BROWSER_ACTION_REFRESH_PAGE = "refreshPage"; 
    public static final String BROWSER_ACTION_NEXT_PAGE = "nextPage"; 

    // WebContent session data 
    public static final String HISTORY = "webcontent.history";
    public static final String HTTP_STATE = "webcontent.http.state";

    protected final static Logger log = LoggerFactory.getLogger(WebContentPortlet.class);

    public static final String FORM_MULTIPART_METHOD = "multipart";

    public static final String NO_URL = "<p>URL source not specified. Go to edit mode and specify an URL.</p>";

    private RewriterController rewriterController;
    private Ruleset rewriterRuleSet;

    private String defaultProxyHost;
    private int defaultProxyPort = -1;

    public WebContentPortlet()
    {
        super();
    }

    /**
     * Initialize portlet configuration.
     */
    public void init(PortletConfig config) throws PortletException
    {
        super.init(config);

        defaultProxyHost = StringUtils.trim(config.getInitParameter("PROXYHOST"));
        defaultProxyPort = NumberUtils.toInt(config.getInitParameter("PROXYPORT"), -1);
    }

    /**
     * processAction() Checks action initiated by the WebContent portlet which
     * means that a user has clicked on an URL
     * 
     * @param actionRequest
     * @param actionResponse
     * @throws PortletException
     * @throws IOException
     */
    public void processAction(ActionRequest actionRequest, ActionResponse actionResponse) throws PortletException,
            IOException
    {
        // check to see if it is a meta-navigation command
        String browserAction = actionRequest.getParameter(BROWSER_ACTION_PARAM);

        if (browserAction != null)
        {
            if (!browserAction.equalsIgnoreCase(BROWSER_ACTION_REFRESH_PAGE))
            {
                // for Refresh, there is nothing special to do - current history page will be re-displayed
                WebContentPageHistory history = (WebContentPageHistory)PortletMessaging.receive(actionRequest, HISTORY);

                if (browserAction.equalsIgnoreCase(BROWSER_ACTION_PREVIOUS_PAGE))
                {
                    if (history.hasPreviousPage())
                    {
                        history.getPreviousPage();
                    }
                }
                else if (browserAction.equalsIgnoreCase(BROWSER_ACTION_NEXT_PAGE))
                {
                    if (history.hasNextPage())
                    {
                        history.getNextPage();
                    }
                }
            }

            return ;   // proceed to doView() with adjusted history
        }

        // Check if an action parameter was defined        
        String webContentURL = actionRequest.getParameter(WebContentRewriter.ACTION_PARAMETER_URL);
        String webContentMethod = StringUtils.defaultIfEmpty(actionRequest.getParameter(WebContentRewriter.ACTION_PARAMETER_METHOD), HttpGet.METHOD_NAME);
        Map webContentParams = new HashMap(actionRequest.getParameterMap()) ;

        // parameter map includes the URL (as ACTION_PARAMETER_URL), but all actual params as well
        webContentParams.remove(WebContentRewriter.ACTION_PARAMETER_URL);
        webContentParams.remove(WebContentRewriter.ACTION_PARAMETER_METHOD);

        if (webContentURL == null || actionRequest.getPortletMode() == PortletMode.EDIT)
        {
            processPreferencesAction(actionRequest, actionResponse);
            webContentURL = actionRequest.getPreferences().getValue("SRC", "http://portals.apache.org");
            // parameters are for the EDIT mode form, and should not be propagated to the subsequent GET in doView
            webContentParams.clear();
        }

        /*
         * If the webContentParameter is not empty attach the URL to the session
         */
        if (webContentURL != null && webContentURL.length() > 0)
        {
            // new page visit - make it the current page in the history
            WebContentPageHistory history = (WebContentPageHistory)PortletMessaging.receive(actionRequest, HISTORY);

            if (history == null)
            {
                history = new WebContentPageHistory();
            }

            history.visitPage(new WebContentPage(webContentURL, webContentMethod, webContentParams));
            PortletMessaging.publish(actionRequest, HISTORY, history);
        }
    }

    /**
     * doView Renders the URL in the following order 1) SESSION_PARAMETER
     * 2)cached version 3) defined for preference SRC
     */
    public void doView(RenderRequest request, RenderResponse response) throws PortletException, IOException
    {
        String viewPage = (String) request.getAttribute(PARAM_VIEW_PAGE);

        if (viewPage != null)
        {
            super.doView(request, response);
            return;
        }

        // view the current page in the history
        WebContentPageHistory history = (WebContentPageHistory)PortletMessaging.receive(request, HISTORY);

        if (history == null)
        {
            history = new WebContentPageHistory();
        }

        WebContentPage currentPage = history.getCurrentPage();

        if (currentPage == null)
        {
            String sourceURL = request.getPreferences().getValue("SRC", "");

            if (sourceURL == null)
            {
                response.getWriter().print(NO_URL);
                return;
            }

            currentPage = new WebContentPage(sourceURL);
        }

        byte[] content = null;

        // get content from current page
        response.setContentType("text/html");

        try
        {
            content = doWebContent(currentPage.getMethod(), currentPage.getUrl(), currentPage.getParams(), request, response);
        }
        catch (Throwable t)
        {
            PrintWriter writer = response.getWriter();
            writer.print("Error retrieveing web content:" + t.getMessage());
            return;
        }

        // write the meta-control navigation header
        PrintWriter writer = response.getWriter();
        writer.print("<block>");

        if (history.hasPreviousPage())
        {
            PortletURL prevAction = response.createActionURL() ;
            prevAction.setParameter(BROWSER_ACTION_PARAM, BROWSER_ACTION_PREVIOUS_PAGE);
            writer.print(" [<a href=\"" + prevAction.toString() +"\">Previous Page</a>] ");
        }

        PortletURL refreshAction = response.createActionURL() ;
        refreshAction.setParameter(BROWSER_ACTION_PARAM, BROWSER_ACTION_REFRESH_PAGE);
        writer.print(" [<a href=\"" + refreshAction.toString() +"\">Refresh Page</a>] ");

        if (history.hasNextPage())
        {
            PortletURL nextAction = response.createActionURL() ;
            nextAction.setParameter(BROWSER_ACTION_PARAM, BROWSER_ACTION_NEXT_PAGE);
            writer.print(" [<a href=\"" + nextAction.toString() +"\">Next Page</a>] ");
        }

        writer.print("</block><hr/>");

        ByteArrayInputStream bais = null;

        try
        {
            bais = new ByteArrayInputStream(content);
            IOUtils.copy(new InputStreamReader(bais, "UTF-8"), writer);
        }
        finally
        {
            IOUtils.closeQuietly(bais);
        }

        // done, cache results in the history and save the history
        history.visitPage(currentPage);
        PortletMessaging.publish(request, HISTORY, history);
    }

    public void doEdit(RenderRequest request, RenderResponse response) throws PortletException, IOException
    {
        response.setContentType("text/html");
        doPreferencesEdit(request, response);
    }

    protected byte[] doWebContent(String method, String sourceAttr, Map sourceParams, RenderRequest request, RenderResponse response)
        throws PortletException
    {
        CloseableHttpClient httpClient = null;
        HttpRequestBase httpRequest = null ;

        try
        {
            if (rewriterController == null)
            {
                String webinfDirPath = getPortletContext().getRealPath("/WEB-INF") + "/";
                rewriterController = getRewriterController(webinfDirPath);
            }

            if (rewriterRuleSet == null)
            {
                InputStream is = null;

                try
                {
                    is = getPortletContext().getResourceAsStream("/WEB-INF/conf/default-rewriter-rules.xml");
                    rewriterRuleSet = rewriterController.loadRuleset(is);
                }
                finally
                {
                    IOUtils.closeQuietly(is);
                }
            }

            WebContentRewriter rewriter = (WebContentRewriter) createRewriter(request, rewriterController, rewriterRuleSet);

            // Set the action and base URLs in the rewriter
            PortletURL action = response.createActionURL();
            rewriter.setActionURL(action);
            URL baseURL = new URL(sourceAttr);
            rewriter.setBaseUrl(baseURL.toString());

            // ...file URLs may be used for testing
            if (baseURL.getProtocol().equals("file"))
            {
                Reader reader = new InputStreamReader((InputStream)baseURL.getContent());
                StringWriter writer = new StringWriter();
                rewriter.rewrite(rewriterController.createParserAdaptor("text/html"), reader, writer);
                writer.flush();
                return writer.toString().getBytes();
            }

            // ...set up URL and HttpClient stuff
            CookieStore cookieStore = new BasicCookieStore();
            httpClient = getHttpClient(request, cookieStore) ;
            String methodName = StringUtils.defaultIfEmpty(method, HttpGet.METHOD_NAME);

            if (StringUtils.equalsIgnoreCase(FORM_MULTIPART_METHOD, methodName))
            {
                httpRequest = createHttpRequest(httpClient, methodName, sourceAttr, null, sourceParams, request);
            }
            else
            {
                httpRequest = createHttpRequest(httpClient, methodName, sourceAttr, sourceParams, null, request);
            }

            byte[] result = doPreemptiveAuthentication(httpClient, cookieStore, httpRequest, request, response);

            // ...get, cache, and return the content
            if (result == null) 
            {
                return doHttpWebContent(httpClient, cookieStore, httpRequest, 0, request, response, rewriter);
            }
            else
            {
                return result;
            }
        }
        catch (PortletException pex)
        {
            // already reported
            throw pex;
        }
        catch (Exception ex)
        {
            String msg = "Exception while rewritting HTML content" ;
            log.error(msg,ex);
            throw new PortletException(msg +", Error: " + ex.getMessage());
        }
        finally
        {
            if (httpRequest != null)
            {
                httpRequest.abort();
            }

            if (httpClient != null)
            {
                try
                {
                    httpClient.close();
                }
                catch (IOException ignore)
                {
                }
            }
        }
    }

    protected byte[] doHttpWebContent(CloseableHttpClient httpClient, CookieStore cookieStore, HttpRequestBase httpRequest, int retryCount,
                                      RenderRequest request, RenderResponse response, WebContentRewriter rewriter) throws PortletException
    {
        CloseableHttpResponse httpResponse = null;
        HttpEntity httpEntity = null;
        InputStream is = null;
        BufferedInputStream bis = null;
        Reader reader = null;

        try
        {
            HttpClientContext httpClientContext = getHttpClientContext(request, httpRequest);

            if (httpClientContext == null)
            {
                httpResponse = httpClient.execute(httpRequest);
            }
            else
            {
                httpResponse = httpClient.execute(httpRequest, httpClientContext);
            }

            // TO AVOID NPE when rewriter argument is null (see org.apache.jetspeed.portlets.sso.SSOWebContentPortlet passing null)
            if (rewriter == null)
            {
                return null;
            }

            // ...reset base URL with fully resolved path (e.g. if a directory, path will end with a /, which it may not have in the call to this method)
            rewriter.setBaseUrl( rewriter.getBaseRelativeUrl( httpRequest.getURI().getPath() )) ;

            // ...save updated state
            List<Cookie> cookies = cookieStore.getCookies();

            if (cookies == null)
            {
                cookies = new ArrayList<Cookie>();
            }

            PortletMessaging.publish(request, HTTP_STATE, cookies);

            // ...check for manual redirects
            int responseCode = httpResponse.getStatusLine().getStatusCode();

            if (responseCode >= 300 && responseCode <= 399)
            {
                // redirection that could not be handled automatically!!! (probably from a POST)
                Header locationHeader = httpResponse.getFirstHeader("Location");
                String redirectLocation = locationHeader != null ? locationHeader.getValue() : null ;

                if (redirectLocation != null)
                {
                    // one more time (assume most params are already encoded & new URL is using GET protocol!)
                    return doWebContent( HttpGet.METHOD_NAME, redirectLocation, null, request, response ) ;
                }
                else
                {
                    // The response is a redirect, but did not provide the new location for the resource.
                    throw new PortletException("Redirection code: " + responseCode + ", but with no redirectionLocation set.");
                }
            }
            else if ( responseCode >= 400 )
            {
                if ( responseCode == 401 )
                {
                    if (retryCount++ < 1 && doRequestedAuthentication(httpClient, cookieStore, httpRequest, request, response))
                    {
                        // try again, now that we are authorizied
                        return doHttpWebContent(httpClient, cookieStore, httpRequest, retryCount, request, response, rewriter);
                    }
                    else
                    {
                        // could not authorize
                        throw new PortletException("Site requested authorization, but we are unable to provide credentials");
                    }
                }
                else if (retryCount++ < 3)
                {
                    log.info("WebContentPortlet.doHttpWebContent() - retrying: " + httpRequest.getURI() + ", response code: " + responseCode);
                    // retry
                    return doHttpWebContent(httpClient, cookieStore, httpRequest, retryCount, request, response, rewriter);
                }
                else
                {
                    // bad
                    throw new PortletException("Failure reading: " + httpRequest.getURI() + ", response code: " + responseCode);
                }
            }

            // ...ok - *now* create the input stream and reader
            httpEntity = httpResponse.getEntity();
            is = httpEntity.getContent();
            bis = new BufferedInputStream(is);

            ContentType contentType = ContentType.getOrDefault(httpEntity);
            Charset charset = contentType.getCharset();
            String encoding = StringUtils.defaultIfEmpty((charset != null ? charset.name() : null), "UTF-8");
            reader = new InputStreamReader(bis, encoding);

            ByteArrayOutputStream byteOutputStream = new ByteArrayOutputStream();
            Writer htmlWriter = new OutputStreamWriter(byteOutputStream, encoding);

            // rewrite and flush output
            rewriter.rewrite(rewriterController.createParserAdaptor("text/html"), reader, htmlWriter);
            htmlWriter.flush();

            // Page has been rewritten
            // TODO: Write it to cache
            //System.out.println(new String(byteOutputStream.toByteArray()));
            return byteOutputStream.toByteArray();
        }
        catch (UnsupportedEncodingException ueex)
        {
            throw new PortletException("Encoding not supported. Error: " + ueex);
        }
        catch (RewriterException rwe)
        {
            throw new PortletException("Failed to rewrite HTML page. Error: " + rwe);
        }
        catch (Exception e)
        {
            throw new PortletException("Exception while rewritting HTML page. Error: " + e);
        }
        finally
        {
            IOUtils.closeQuietly(reader);
            IOUtils.closeQuietly(bis);
            IOUtils.closeQuietly(is);

            if (httpEntity != null)
            {
                try
                {
                    EntityUtils.consume(httpEntity);
                }
                catch (Exception ignore)
                {
                }
            }

            if (httpResponse != null)
            {
                try
                {
                    httpResponse.close();
                }
                catch (IOException ignore)
                {
                }
            }
        }
    }

    protected byte[] doPreemptiveAuthentication(CloseableHttpClient client, CookieStore cookieStore, HttpRequestBase httpRequest, RenderRequest request, RenderResponse response)
    {
        // derived class responsibilty - return true, if credentials have been set
        return null ;
    }

    protected boolean doRequestedAuthentication(CloseableHttpClient client, CookieStore cookieStore, HttpRequestBase httpRequest, RenderRequest request, RenderResponse response)
    {
        // derived class responsibilty - return true, if credentials have been set
        return false ;
    }

    /*
     * Generate a rewrite controller using the basic rules file
     */
    protected RewriterController getRewriterController(String contextPath) throws Exception
    {
        Class[] rewriterClasses = new Class[] { WebContentRewriter.class, WebContentRewriter.class };
        Class[] adaptorClasses = new Class[] { NekoParserAdaptor.class, SaxParserAdaptor.class };
        RewriterController rewriterController = 
                        new MappingRewriterController(contextPath + "conf/rewriter-rules-mapping.xml", 
                                                      Arrays.asList(rewriterClasses),
                                                      Arrays.asList(adaptorClasses));
        return rewriterController;
    }

    protected Rewriter createRewriter(PortletRequest request, RewriterController rewriterController, Ruleset ruleset) throws RewriterException
    {
        RulesetRewriter rewriter = rewriterController.createRewriter(ruleset);
        return rewriter;
    }

    protected HttpClientBuilder getHttpClientBuilder(PortletRequest request, CookieStore cookieStore) {
        HttpClientBuilder builder = 
                        HttpClients.custom()
                        .setRedirectStrategy(new LaxRedirectStrategy());

        String proxyHost = StringUtils.trim(request.getPreferences().getValue("PROXYHOST", defaultProxyHost));
        int proxyPort = NumberUtils.toInt(request.getPreferences().getValue("PROXYPORT", Integer.toString(defaultProxyPort)), -1);

        if (!StringUtils.isEmpty(proxyHost))
        {
            if (proxyPort > 0)
            {
                builder.setProxy(new HttpHost(proxyHost, proxyPort));
            }
            else
            {
                builder.setProxy(new HttpHost(proxyHost));
            }
        }

        if (cookieStore != null)
        {
            // reuse existing state, if we have been here before
            List<Cookie> cookies = (List<Cookie>) PortletMessaging.receive(request, HTTP_STATE);

            if (cookies != null)
            {
                for (Cookie cookie : cookies)
                {
                    cookieStore.addCookie(cookie);
                }
            }

            builder.setDefaultCookieStore(cookieStore);
        }

        return builder;
    }

    protected CloseableHttpClient getHttpClient(RenderRequest request, CookieStore cookieStore) throws IOException
    {
        HttpClientBuilder builder = getHttpClientBuilder(request, cookieStore);
        return builder.build();
    }

    /**
     * Override this method to give a custom <code>HttpClientContext</code>
     * when executing <code>HttpClient</code>.
     * @param request
     * @return
     */
    protected HttpClientContext getHttpClientContext(PortletRequest request, HttpRequestBase httpRequest)
    {
        return null;
    }

    protected HttpRequestBase createHttpRequest(CloseableHttpClient client, String method, String uri, Map<String, String []> queryParams, Map<String, String []> formPostParams, RenderRequest request) throws IOException, URISyntaxException
    {
        // formMethod = FORM_MULTIPART_METHOD;
        HttpRequestBase httpRequest = null;

        URIBuilder uriBuilder = new URIBuilder(uri);

        if (queryParams != null)
        {
            String name = null;
            String [] values = null;

            for (Map.Entry<String, String[]> entry : queryParams.entrySet())
            {
                name = entry.getKey();
                values = entry.getValue();

                if (!StringUtils.isEmpty(name) && values != null)
                {
                    for (String value : values)
                    {
                        uriBuilder.addParameter(name, value);
                    }
                }
            }
        }

        if (StringUtils.equalsIgnoreCase(HttpPost.METHOD_NAME, method))
        {
            httpRequest = new HttpPost(uriBuilder.build());

            if (formPostParams != null)
            {
                List<NameValuePair> formParams = new ArrayList<NameValuePair>();

                String name = null;
                String [] values = null;

                for (Map.Entry<String, String[]> entry : formPostParams.entrySet())
                {
                    name = entry.getKey();
                    values = entry.getValue();

                    if (!StringUtils.isEmpty(name) && values != null)
                    {
                        for (String value : values)
                        {
                            formParams.add(new BasicNameValuePair(name, value));
                        }
                    }
                }

                if (!formParams.isEmpty())
                {
                    UrlEncodedFormEntity httpEntity = new UrlEncodedFormEntity(formParams, Consts.UTF_8);
                    ((HttpPost) httpRequest).setEntity(httpEntity);
                }
            }
        }
        else if (StringUtils.equalsIgnoreCase(HttpGet.METHOD_NAME, method) || StringUtils.isBlank(method))
        {
            httpRequest = new HttpGet(uriBuilder.build());
        }

        // propagate User-Agent, so target site does not think we are a D.O.S.
        // attack
        String userAgentHeaderValue = request.getProperty("User-Agent");
        httpRequest.setHeader("User-Agent", userAgentHeaderValue);

        return httpRequest;
    }
}
