/*
Licensed to the Apache Software Foundation (ASF) under one or more
contributor license agreements.  See the NOTICE file distributed with
this work for additional information regarding copyright ownership.
The ASF licenses this file to You under the Apache License, Version 2.0
(the "License"); you may not use this file except in compliance with
the License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package tasks

import (
	"encoding/json"
	"fmt"
	"github.com/apache/incubator-devlake/core/dal"
	"github.com/apache/incubator-devlake/core/errors"
	"github.com/apache/incubator-devlake/core/models/domainlayer/didgen"
	"github.com/apache/incubator-devlake/core/models/domainlayer/ticket"
	"github.com/apache/incubator-devlake/core/plugin"
	"github.com/apache/incubator-devlake/helpers/pluginhelper/api"
	"github.com/apache/incubator-devlake/plugins/tapd/models"
	"io"
	"net/http"
	"net/url"
	"reflect"
	"strconv"
	"strings"
)

type Page struct {
	Data Data `json:"data"`
}
type Data struct {
	Count int `json:"count"`
}

var priorityMap = map[string]string{
	"1": "Nice To Have",
	"2": "Low",
	"3": "Middle",
	"4": "High",
}

var accountIdGen *didgen.DomainIdGenerator
var workspaceIdGen *didgen.DomainIdGenerator
var iterIdGen *didgen.DomainIdGenerator

func getAccountIdGen() *didgen.DomainIdGenerator {
	if accountIdGen == nil {
		accountIdGen = didgen.NewDomainIdGenerator(&models.TapdAccount{})
	}
	return accountIdGen
}

func getWorkspaceIdGen() *didgen.DomainIdGenerator {
	if workspaceIdGen == nil {
		workspaceIdGen = didgen.NewDomainIdGenerator(&models.TapdWorkspace{})
	}
	return workspaceIdGen
}

func getIterIdGen() *didgen.DomainIdGenerator {
	if iterIdGen == nil {
		iterIdGen = didgen.NewDomainIdGenerator(&models.TapdIteration{})
	}
	return iterIdGen
}

// res will not be used
func GetTotalPagesFromResponse(r *http.Response, args *api.ApiCollectorArgs) (int, errors.Error) {
	data := args.Ctx.GetData().(*TapdTaskData)
	apiClient, err := NewTapdApiClient(args.Ctx.TaskContext(), data.Connection)
	if err != nil {
		return 0, err
	}
	query := url.Values{}
	query.Set("workspace_id", fmt.Sprintf("%v", data.Options.WorkspaceId))
	res, err := apiClient.Get(fmt.Sprintf("%s/count", r.Request.URL.Path), query, nil)
	if err != nil {
		return 0, err
	}
	var page Page
	err = api.UnmarshalResponse(res, &page)

	count := page.Data.Count
	totalPage := count/args.PageSize + 1

	return totalPage, err
}

func parseIterationChangelog(taskCtx plugin.SubTaskContext, old string, new string) (iterationFromId uint64, iterationToId uint64, err errors.Error) {
	data := taskCtx.GetData().(*TapdTaskData)
	db := taskCtx.GetDal()
	iterationFrom := &models.TapdIteration{}
	clauses := []dal.Clause{
		dal.From(&models.TapdIteration{}),
		dal.Where("connection_id = ? and workspace_id = ? and name = ?",
			data.Options.ConnectionId, data.Options.WorkspaceId, old),
	}
	err = db.First(iterationFrom, clauses...)
	if err != nil && !db.IsErrorNotFound(err) {
		return 0, 0, err
	}

	iterationTo := &models.TapdIteration{}
	clauses = []dal.Clause{
		dal.From(&models.TapdIteration{}),
		dal.Where("connection_id = ? and workspace_id = ? and name = ?",
			data.Options.ConnectionId, data.Options.WorkspaceId, new),
	}
	err = db.First(iterationTo, clauses...)
	if err != nil && !db.IsErrorNotFound(err) {
		return 0, 0, err
	}
	return iterationFrom.Id, iterationTo.Id, nil
}

func GetRawMessageDirectFromResponse(res *http.Response) ([]json.RawMessage, errors.Error) {
	body, err := io.ReadAll(res.Body)
	res.Body.Close()
	if err != nil {
		return nil, errors.Convert(err)
	}
	return []json.RawMessage{body}, nil
}

func GetRawMessageArrayFromResponse(res *http.Response) ([]json.RawMessage, errors.Error) {
	var data struct {
		Data []json.RawMessage `json:"data"`
	}
	err := api.UnmarshalResponse(res, &data)
	return data.Data, err
}

type TapdApiParams struct {
	ConnectionId uint64
	CompanyId    uint64
	WorkspaceId  uint64
}

func CreateRawDataSubTaskArgs(taskCtx plugin.SubTaskContext, rawTable string, useCompanyId bool) (*api.RawDataSubTaskArgs, *TapdTaskData) {
	data := taskCtx.GetData().(*TapdTaskData)
	filteredData := *data
	filteredData.Options = &TapdOptions{}
	*filteredData.Options = *data.Options
	var params = TapdApiParams{
		ConnectionId: data.Options.ConnectionId,
		WorkspaceId:  data.Options.WorkspaceId,
	}
	if data.Options.CompanyId != 0 && useCompanyId {
		params.CompanyId = data.Options.CompanyId
	} else {
		filteredData.Options.CompanyId = 0
	}
	rawDataSubTaskArgs := &api.RawDataSubTaskArgs{
		Ctx:    taskCtx,
		Params: params,
		Table:  rawTable,
	}
	return rawDataSubTaskArgs, &filteredData
}

// getTapdTypeMappings will map story types in _tool_tapd_workitem_types to our typeMapping
func getTapdTypeMappings(data *TapdTaskData, db dal.Dal, system string) (map[uint64]string, errors.Error) {
	typeIdMapping := make(map[uint64]string)
	issueTypes := make([]models.TapdWorkitemType, 0)
	clauses := []dal.Clause{
		dal.From(&models.TapdWorkitemType{}),
		dal.Where("connection_id = ? and workspace_id = ? and entity_type = ?",
			data.Options.ConnectionId, data.Options.WorkspaceId, system),
	}
	err := db.All(&issueTypes, clauses...)
	if err != nil {
		return nil, err
	}
	for _, issueType := range issueTypes {
		typeIdMapping[issueType.Id] = issueType.Name
	}
	return typeIdMapping, nil
}

func getStdTypeMappings(data *TapdTaskData) map[string]string {
	stdTypeMappings := make(map[string]string)
	for userType, stdType := range data.Options.TransformationRules.TypeMappings {
		stdTypeMappings[userType] = strings.ToUpper(stdType.StandardType)
	}
	return stdTypeMappings
}

func getStatusMapping(data *TapdTaskData) map[string]string {
	statusMapping := make(map[string]string)
	mapping := data.Options.TransformationRules.StatusMappings
	for std, orig := range mapping {
		for _, v := range orig {
			statusMapping[v] = std
		}
	}

	return statusMapping
}

func getDefaultStdStatusMapping[S models.TapdStatus](data *TapdTaskData, db dal.Dal, statusList []S) (map[string]string, func(statusKey string) string, errors.Error) {
	clauses := []dal.Clause{
		dal.Where("connection_id = ? and workspace_id = ?", data.Options.ConnectionId, data.Options.WorkspaceId),
	}
	err := db.All(&statusList, clauses...)
	if err != nil {
		return nil, nil, err
	}

	statusLanguageMap := make(map[string]string, len(statusList))
	statusLastStepMap := make(map[string]bool, len(statusList))

	for _, v := range statusList {
		statusLanguageMap[v.GetEnglish()] = v.GetChinese()
		statusLastStepMap[v.GetChinese()] = v.GetIsLastStep()
	}
	getStdStatus := func(statusKey string) string {
		if statusLastStepMap[statusKey] {
			return ticket.DONE
		} else if statusKey == "草稿" {
			return ticket.TODO
		} else {
			return ticket.IN_PROGRESS
		}
	}
	return statusLanguageMap, getStdStatus, nil
}

func unicodeToZh(s string) (string, error) {
	// strconv.Quote(s) will add additional `"`, so we need to do `Unquote` again
	str, err := strconv.Unquote(strings.Replace(strconv.Quote(s), `\\u`, `\u`, -1))
	if err != nil {
		return "", err
	}
	return str, nil
}

func convertUnicode(p interface{}) errors.Error {
	var err errors.Error
	pType := reflect.TypeOf(p)
	if pType.Kind() != reflect.Ptr {
		panic("expected a pointer to a struct")
	}
	pValue := reflect.ValueOf(p).Elem()
	if pValue.Kind() != reflect.Struct {
		panic("expected a pointer to a struct")
	}
	after, err := errors.Convert01(unicodeToZh(pValue.FieldByName("ValueAfterParsed").String()))
	if err != nil {
		return err
	}
	before, err := errors.Convert01(unicodeToZh(pValue.FieldByName("ValueBeforeParsed").String()))
	if err != nil {
		return err
	}
	if after == "--" {
		after = ""
	}
	if before == "--" {
		before = ""
	}
	// set ValueAfterParsed
	valueAfterField := pValue.FieldByName("ValueAfterParsed")
	valueAfterField.SetString(after)
	valueBeforeField := pValue.FieldByName("ValueBeforeParsed")
	valueBeforeField.SetString(before)
	return nil
}

// replace ";" with ","
func replaceSemicolonWithComma(str string) string {
	res := strings.ReplaceAll(str, ";", ",")
	return strings.TrimRight(res, ",")
}

// generate domain account id for each user
// param is a string with format "user1,user2,user3"
func generateDomainAccountIdForUsers(param string, connectionId uint64) string {
	if param == "" {
		return ""
	}
	param = replaceSemicolonWithComma(param)
	users := strings.Split(param, ",")
	var res []string
	for _, user := range users {
		res = append(res, getAccountIdGen().Generate(connectionId, user))
	}
	return strings.Join(res, ",")
}
